C Modified random walk simulation (one iteration) of pedigree missing
C genotypes (Metropolis-Hastings algorithm) -- proposal uses founder
C allele mutations propagated throughout then pedigree conditional
C on (identity by) descent, swapping ibd origins for heterozygotes,
C alternated with a randomization of descent conditional on marker
C genotype. This procedure has the advantage of being quick,
C but the proposal probabilities are not always symmetric, so they
C are combined with additional local proposals
C
C It is alternated with a locally updating Gibbs sampler. This jointly
C simulates Untyped x Untyped founder matings genotypes conditional on
C offspring and other spouses; other genotypes individual-by-individual,
C conditional on parental, spouse and child genotypes.
C
      subroutine drop(it,pedigree,num,nfound,id,fa,mo,nummat,
     2                cntmat,untyped,numal,gfrq,set,set2,sibd,
     3                key, iprop, plevel)
      integer MAXALL, MAXG, MAXSIZ
      parameter(MAXALL=60, MAXG=MAXALL*(MAXALL+1)/2,
     &          MAXSIZ=1000)

      integer iprop, it, plevel
C pedigree structure
      character*10 pedigree
      integer num, nfound
      character*10 id(MAXSIZ)
      integer fa(MAXSIZ), mo(MAXSIZ)
C genotypic arrays for pedigree: current, proposed; resulting ibd status
      integer set(MAXSIZ,2), set2(MAXSIZ,2)
      integer sibd(MAXSIZ,2), key(2*MAXSIZ)
C indicator of untyped individual
      logical untyped(MAXSIZ)
C allele frequencies structure
      integer numal
      double precision gfrq(MAXG)
C number of untyped matings -- used to decide number and type of mutations
      integer nummat, cntmat(MAXALL,2)
C local variables
      integer gibbsit, gprop, i, mat, par1, par2
      logical xmale
      double precision lr, qa 
C functions
      integer irandom, parcon

      gibbsit=5
      xmale=.false.
      iprop=irandom(1,60)
      if (iprop.gt.4) iprop=4
C
C Metropolis multisite proposals
      if (iprop.le.3) then
C
C Do ibd dropping
C
        if (iprop.eq.1) then
          do 30 i=1,num
            set2(i,1)=set(i,1)
            set2(i,2)=set(i,2)
   30     continue
          call fsimped(it,pedigree,num,nfound,id,fa,mo,set2,sibd,
     &                 untyped,key,plevel)
C
C Or mutate founder allele(s)
C
        else if (iprop.eq.2) then
          call simibd(2,pedigree,num,nfound,fa,mo,set,sibd)
          call mutate(numal,num,nfound,set,sibd,set2,untyped)
C
C Or switch parents of origin
C
        else
          call simibd(2,pedigree,num,nfound,fa,mo,set,sibd)
          call switch(num,nfound,id,fa,mo,set,sibd,set2,untyped)
        end if
C Further shuffle genotypes -- here untyped founder matings
        do 20 mat=1,nummat
          par1=cntmat(mat,1)
          par2=cntmat(mat,2)
          call simpar(num,nfound,fa,mo,set,par1,par2,numal)
   20   continue
C
C now check if the proposal is an acceptable one
C
        if (plevel.gt.2) then
          do 999 i=1,num
            write(*,*) it, ' ', id(i), set(i,1), set(i,2), ' -> ',
     &        set2(i,1), set2(i,2), ' {',sibd(i,1),sibd(i,2),' }'
  999     continue
        end if
        lr=1.0d0
        do 100 i=1,nfound
        if (untyped(i)) then
          lr=lr*gfrq(set2(i,2)*(set2(i,2)-1)/2+set2(i,1))/
     &          gfrq(set(i,2)*(set(i,2)-1)/2+set(i,1))
        end if
  100   continue
        do 110 i=nfound+1,num
          par2=parcon(set2(i,1),set2(i,2),set2(fa(i),1),set2(fa(i),2),
     &                set2(mo(i),1),set2(mo(i),2),xmale)
          par1=parcon(set(i,1),set(i,2),set(fa(i),1),set(fa(i),2),
     &                set(mo(i),1),set(mo(i),2),xmale)
          if (par1.ne.par2) then
            lr=lr*dfloat(par2)/dfloat(par1)
          end if
  110   continue
        qa=min(1.0d0,lr)
C
C If accepted, update genotypes
C
        if (qa.gt.random()) then
          if (plevel.gt.2) then
            write(*,'(a,i1,a,f12.4)') 
     &        'Proposal type ',iprop,' accepted ',lr
          end if
          do 150 i=1,num
          if (untyped(i)) then
            set(i,1)=set2(i,1)
            set(i,2)=set2(i,2)
          end if
  150     continue
        else 
          if (plevel.gt.2) then
            write(*,'(a,i1,a,f12.4)') 
     &        'Proposal type ',iprop,' rejected ',lr
          end if
          iprop=-iprop
        end if
      end if
C
C now local updating via Gibbs sampler
C
      gprop=0
      do 200 mat=1, nummat
        gprop=gprop+1
        key(gprop)=MAXSIZ+mat
  200 continue
      do 250 i=1, num
      if (untyped(i)) then
        gprop=gprop+1
        key(gprop)=i
      end if
  250 continue
      call permut(gprop, key)

      do 300 j=1, gibbsit
      do 300 i=1, gprop
        if (key(i).gt.MAXSIZ) then
C Untyped x untyped founder mating update
          mat=key(i)-MAXSIZ
          par1=cntmat(mat,1)
          par2=cntmat(mat,2)
          call simnuc(par1,par2,num,nfound,fa,mo,set,numal,gfrq)
        else
C individual update
          call simnuc(key(i),key(i),num,nfound,fa,mo,set,numal,gfrq)
        end if
  300 continue
      call simibd(2,pedigree,num,nfound,fa,mo,set,sibd)

      if (plevel.gt.1) then
        write(*,'(a,i6)') 
     &    'Simulated missing genotypes for iteration ',it
        do 160 i=1,num
        if (untyped(i)) then
          write(*,'(i5,1x,a10,2(1x,i3))') it,id(i),set(i,1),set(i,2)
        end if
  160   continue
      end if
      return
      end
C end-of-drop
C 
C  Gibbs sampler for codominant marker locus
C  Simulate parental genotypes for untyped x untyped
C  mating conditional on offspring genotypes
C  or for untyped nonfounders conditional on offspring, spouses 
C  and parents. Family may be contained within larger pedigree, and
C  multiple spouses are possible
C
      subroutine simnuc(par1,par2,num,nfound,fa,mo,set,numal,gfrq)
C
C  Pedigree structure
      integer MAXALL, MAXG, MAXSIZ, MISS
      parameter(MAXALL=60, MAXG=MAXALL*(MAXALL+1)/2,
     &          MAXSIZ=1000, MISS=-9999)
      integer num, nfound, fa(MAXSIZ),mo(MAXSIZ), set(MAXSIZ,2)
      integer par1, par2
C genotype frequencies structure
      integer numal
      double precision gfrq(MAXG)
C local variables
      integer fin, i, sta
      double precision targt
C functions
      real random

      sta=MISS
      fin=MISS
      do 10 i=nfound+1,num
      if (par1.eq.fa(i) .or. par2.eq.mo(i)) then
        if (sta.eq.MISS) then
          sta=i
          fin=i
        elseif (i.gt.fin) then
          fin=i
        end if
      end if
   10 continue
C
      if (par1.ne.par2) then
        targt=1.0d0
        call nuclik(par1,par2,sta,fin,fa,mo,set,
     &              numal,gfrq,targt)
        targt=dble(random())*targt
        call nuclik(par1,par2,sta,fin,fa,mo,set,
     &              numal,gfrq,targt)
      else if (par1.le.nfound) then
        targt=1.0d0
        call foulik(par1,fa,mo,set,sta,fin,numal, gfrq, targt)
        targt=dble(random())*targt
        call foulik(par1,fa,mo,set,sta,fin,numal, gfrq, targt)
      else 
        targt=1.0d0
        call onelik(par1,fa,mo,set,sta,fin,targt)
        targt=dble(random())*targt
        call onelik(par1,fa,mo,set,sta,fin,targt)
      end if
      return
      end
C end-of-simnuc
C
C Nuclik is run twice, once to calculate the total likelihood <totlik> of the
C legal genotypes (with target=1), the second time to select a 
C parental genotypes with target ~ U(0,totlik).
C
      subroutine nuclik(par1,par2,sta,fin,fa,mo,set,
     &                  numal,gfrq,targt)
      integer MAXALL, MAXG, MAXSIZ
      parameter(MAXALL=60, MAXG=MAXALL*(MAXALL+1)/2,
     &          MAXSIZ=1000)
      double precision targt
C  Pedigree structure
      integer fa(MAXSIZ),mo(MAXSIZ), set(MAXSIZ,2)
      integer fin, g1, g2, mg1, mg2, par1, par2, pg1, pg2, sta
C genotype frequencies structure
      integer numal
      double precision gfrq(MAXG)
C local variables
      integer con, i
      logical xmale
      double precision lik, totlik
C functions
      integer parcon
C 
      xmale=.false.
      totlik=0.0d0

      g1=0
      do 5 mg2=1,numal
      do 5 mg1=1,mg2
        g1=g1+1
        lik=gfrq(g1)
        g2=0
        do 6 pg2=1,numal
        do 6 pg1=1,pg2
          g2=g2+1
          lik=gfrq(g1)*gfrq(g2)
C         if (mg1.ne.mg2) lik=lik+lik
C         if (pg1.ne.pg2) lik=lik+lik
          set(par1,1)=pg1
          set(par1,2)=pg2
          set(par2,1)=mg1
          set(par2,2)=mg2
          do 10 i=sta,fin
            con=parcon(set(i,1),set(i,2),set(fa(i),1),set(fa(i),2),
     &                 set(mo(i),1),set(mo(i),2),xmale)
            if ((par1.eq.fa(i) .or. par2.eq.mo(i)) .and. con.eq.0) then
              goto 55
            end if
            lik=lik*0.25d0*dfloat(con)
   10     continue
C
C else (if consistent) add to legal genotypes
C
          totlik=totlik+lik
C
C check to see if have selected current parental genotypes
C
            if (totlik.ge.targt) then
              return
            end if
   55     continue
    6   continue
    5 continue
      targt=totlik
      return
      end
C end-of-nuclik
C
C Founder codominant locus conditional likelihood
C foulik is run twice, once to calculate the total likelihood <totlik> of the
C legal genotypes (with target=1), the second time to select a genotype,
C with target ~ U(0,totlik).
C
      subroutine foulik(idx, fa, mo, set, sta, fin, numal, gfrq, targt)
      integer MAXALL, MAXG, MAXSIZ, MISS
      parameter(MAXALL=60, MAXG=MAXALL*(MAXALL+1)/2,
     &          MAXSIZ=1000, MISS=-9999)
      integer idx
      double precision targt
C  Pedigree structure
      integer fa(MAXSIZ), mo(MAXSIZ), set(MAXSIZ,2)
      integer fin, sta
C genotype frequencies structure
      integer numal
      double precision gfrq(MAXG)
C local variables
      integer con, g1, g2, ng, i
      logical xmale
      double precision lik, totlik
C functions
      integer parcon
C 
      xmale=.false.
      totlik=0.0d0
      ng=0
      do 5 g1=1,numal
      do 5 g2=1,g1 
        ng=ng+1
        lik=gfrq(ng)
C       if (g1.ne.g2) lik=lik+lik
        set(idx,1)=g2
        set(idx,2)=g1
        if (sta.ne.MISS) then
          do 10 i=sta,fin
            con=parcon(set(i,1),set(i,2),set(fa(i),1),set(fa(i),2),
     &                 set(mo(i),1),set(mo(i),2),xmale)
            if ((idx.eq.fa(i) .or. idx.eq.mo(i)) .and. con.eq.0) then
              goto 55
            end if
            lik=lik*0.25d0*dfloat(con)
   10     continue
        end if
        totlik=totlik+lik
C exit if target reached
        if (totlik.ge.targt) then
          return
        end if
   55   continue
    5 continue
      targt=totlik
      return
      end
C end-of-foulik
C
C nonfounder codominant locus conditional likelihood
C onelik is run twice, once to calculate the total likelihood <totlik> of the
C legal genotypes (with target=1), the second time to select a genotype,
C with target ~ U(0,totlik).
C
      subroutine onelik(idx,fa,mo,set,sta,fin,targt)
      integer MAXSIZ, MISS
      parameter(MAXSIZ=1000, MISS=-9999)
      integer idx
      double precision targt
C  Pedigree structure
      integer fa(MAXSIZ), mo(MAXSIZ), set(MAXSIZ,2)
      integer fin, sta
C local variables
      integer con, g1, g2, i, i1, i2
      logical xmale
      double precision lik, totlik
C functions
      integer parcon
C 
      xmale=.false.
      totlik=0.0d0
      do 5 i1=1,2
      do 5 i2=1,2
        lik=0.25d0
        g1=set(fa(idx),i1)
        g2=set(mo(idx),i2)
        call order(g1,g2)
        set(idx,1)=g1
        set(idx,2)=g2
        if (sta.ne.MISS) then
          do 10 i=sta,fin
            con=parcon(set(i,1),set(i,2),set(fa(i),1),set(fa(i),2),
     &                 set(mo(i),1),set(mo(i),2),xmale)
            if ((idx.eq.fa(i) .or. idx.eq.mo(i)) .and. con.eq.0) then
              goto 55
            end if
            lik=lik*0.25d0*dfloat(con)
   10     continue
        end if
        totlik=totlik+lik
C exit if target reached
        if (totlik.ge.targt) then
          return
        end if
   55   continue
    5 continue
      targt=totlik
      return
      end
C end-of-onelik
C 
C  Propose parental genotypes for untyped x untyped
C  mating conditional on offspring genotypes
C  Family may be contained within larger pedigree, and
C  multiple spouses are possible
C
      subroutine simpar(num,nfound,fa,mo,set,par1,par2,numal)
C
C  Pedigree structure
      integer MAXALL, MAXG, MAXSIZ, MISS
      parameter(MAXALL=40, MAXG=MAXALL*(MAXALL+1)/2,
     &          MAXSIZ=1000, MISS=-9999)
      integer num, nfound, fa(MAXSIZ),mo(MAXSIZ), set(MAXSIZ,2)
      integer par1, par2
C local variables
      integer fin, i, sta
      integer targt, totp
C functions
      integer irandom

      sta=MISS
      fin=MISS
      do 10 i=nfound+1,num
      if (par1.eq.fa(i) .or. par2.eq.mo(i)) then
        if (sta.eq.MISS) then
          sta=i
          fin=i
        elseif (i.gt.fin) then
          fin=i
        end if
      end if
   10 continue
C
      targt=MAXG*MAXG
      call inuclik(fa,mo,set,par1,par2,sta,fin,
     &            numal,targt,totp)
      targt=irandom(1,totp)
      call inuclik(fa,mo,set,par1,par2,sta,fin,
     &            numal,targt,totp)
C
      return
      end
C end-of-simpar
C
C inuclik is an integer version of nuclik
C inuclik is run twice, once to calculate the total number <totp> of the
C legal genotypes (with target=MAXG*MAXG), the second time to select a 
C parental genotype.
C
      subroutine inuclik(fa,mo,set,par1,par2,sta,fin,
     &                   numal,targt,totp)
      integer MAXSIZ
      parameter(MAXSIZ=1000)
      integer targt, totp
C  Pedigree structure
      integer fa(MAXSIZ),mo(MAXSIZ), set(MAXSIZ,2)
      integer fin, mg1, mg2, par1, par2, pg1, pg2, sta
C allele frequencies structure
      integer numal

C local variables
      integer con, i
      logical xmale
C functions
      integer parcon
C 
      xmale=.false.
      totp=0

      do 5 mg2=1,numal
      do 5 mg1=1,mg2
      do 5 pg2=1,numal
      do 5 pg1=1,pg2
        set(par1,1)=pg1
        set(par1,2)=pg2
        set(par2,1)=mg1
        set(par2,2)=mg2
        do 10 i=sta,fin
          con=parcon(set(i,1),set(i,2),set(fa(i),1),set(fa(i),2),
     &               set(mo(i),1),set(mo(i),2),xmale)
          if ((par1.eq.fa(i) .or. par2.eq.mo(i)) .and. con.eq.0) then
            goto 55
          end if
   10   continue
C
C else (if consistent) add to legal genotypes
C
        totp=totp+1
C
C check to see if have selected current parental genotypes
C
          if (totp.eq.targt) then
            return
          end if
   55   continue
    5 continue
      return
      end
C end-of-inuclik
C
C Mutate 1-4 allele in untyped founders.  Can never fail, due "backmutation".
C
C (1) mutate an allele never transmitted to a typed individual
C (2) swap parent of origin if have no offspring sharing ibd-allele
C
      subroutine mutate(numal,num,nfound,set,sibd,set2,untyped)
      integer MAXSIZ
      parameter (MAXSIZ=1000)
      integer numal
C Pedigree structure
      integer num, nfound
      integer set(MAXSIZ,2), sibd(MAXSIZ,2), set2(MAXSIZ,2)
      logical untyped(MAXSIZ)
C
C mut designates ibd-allele, prop the corresponding marker allele
      integer nmut
      integer mut(4), prop(4)
      integer g1,g2,i,j
C functions
      integer irandom

    1 continue

        nmut=irandom(1,4)
        do 5 j=1,nmut
          mut(j)=irandom(1,2*nfound)
          prop(j)=irandom(1,numal)
    5    continue
         do 10 i=1,num
           g1=set(i,1)
           g2=set(i,2)
           do 15 j=1, nmut
             if (sibd(i,1).eq.mut(j)) g1=prop(j)
             if (sibd(i,2).eq.mut(j)) g2=prop(j)
   15      continue
           call order(g1,g2)
C midloop break
      if ( .not.untyped(i) .and.
     &     (set(i,1).ne.g1 .or. set(i,2).ne.g2)) then
        goto 1
      end if
C
          call update(i,g1,g2,set2)
   10   continue
      return
      end
C end-of-mutate
C
C Do a switch of grandparent of origin of alleles = switch lineage
C
      subroutine switch(num,nfound,id,fa,mo,set,sibd,set2,untyped)
      integer KNOWN,MAXSIZ,MISS
      parameter (KNOWN=0,MAXSIZ=1000,MISS=-9999)
C Pedigree structure
      character*10 id(MAXSIZ)
      integer num, nfound, fa(MAXSIZ), mo(MAXSIZ)
      integer set(MAXSIZ,2), sibd(MAXSIZ,2), set2(MAXSIZ,2)
      logical untyped(MAXSIZ)
C
      integer g(2), i, idx, ntrials, par
      logical fin, xmale

C functions
      integer irandom, parcon

      xmale=.false.
      ntrials=0

    1 continue
C
      ntrials=ntrials+1
C
C give up if <num> unsuccessful trials
C
      if (ntrials.gt.num) then
        do 220 i=1,num
          set2(i,1)=set(i,1)
          set2(i,2)=set(i,2)
  220   continue
        return
      end if
C
C provide a heterozygote candidate with both parents untyped
C
      idx=irandom(nfound+1,num)

      if (set(idx,1).eq.set(idx,2) .or. 
     &     .not.untyped(fa(idx)) .or. .not.untyped(mo(idx))) then
         goto 1
      end if
C 
      do 10 i=1,num
        set2(i,1)=MISS
        set2(i,2)=MISS
   10 continue
      g(2)=set(idx,1)
      g(1)=set(idx,2)
      set2(idx,1)=1
      set2(idx,2)=2
C
C Each iteration moves as far up the pedigree as possible 
C
   15 continue
        fin=.true.
        do 100 i=nfound+1,num
        if (set2(i,1).ne.MISS) then
          par=fa(i)
          if (set2(par,1).eq.MISS .and. untyped(par)) then
            call cpibd(par,i,sibd,set2,KNOWN)
            fin=.false.
          end if
          par=mo(i)
          if (set2(par,1).eq.MISS .and. untyped(par)) then
            call cpibd(par,i,sibd,set2,KNOWN)
            fin=.false.
          end if
        end if
  100   continue
      if (.not.fin) goto 15
C
C See if reached one (loop) or two untyped founders, so swap feasible
C
      par=0
      do 110 i=1,nfound
      if (set2(i,1).gt.KNOWN .or. set2(i,2).gt.KNOWN) then
        par=par+1
      end if
  110 continue
C       write(*,*) 'idx=',id(idx), ' par=', par
C       do 999 i=1,num
C         write(*,*) id(i), ' ', set(i,1), set(i,2), ' -> ',
C    &      set2(i,1), set2(i,2), ' {',sibd(i,1),sibd(i,2),'}'
C 999   continue
      if (par.eq.0) then
        goto 1
      end if
C
C swap appropriate alleles of simulated genotype
C
      do 150 i=1,num
        if (set2(i,1).gt.KNOWN) set2(i,1)=g(set2(i,1))
        if (set2(i,2).gt.KNOWN) set2(i,2)=g(set2(i,2))
        if (set2(i,1).le.KNOWN) set2(i,1)=set(i,1)
        if (set2(i,2).le.KNOWN) set2(i,2)=set(i,2)
        call order(set2(i,1),set2(i,2))
        if (i.gt.nfound .and. parcon(set2(i,1),set2(i,2),
     2      set2(fa(i),1),set2(fa(i),2),set2(mo(i),1),
     3      set2(mo(i),2), xmale).eq.0) then
          goto 1
        end if
  150 continue
      return
      end
C end-of-switch
C
C Copy ibd for a pair of relatives.
C Person j has a typing-genotype at sibd2, person i does not.
C The typing-allele corresponding to that shared at sibd()
C is "transmitted" to person i from person j.
C
      subroutine cpibd(i,j,sibd,sibd2,imiss)
      integer MAXSIZ, MISS
      parameter(MAXSIZ=1000,MISS=-9999)
      integer i,imiss,j
      integer sibd(MAXSIZ,2), sibd2(MAXSIZ,2)

      if (sibd(i,1).eq.sibd(j,1)) then
        sibd2(i,1)=sibd2(j,1)
      end if
      if (sibd(i,1).eq.sibd(j,2)) then
        sibd2(i,1)=sibd2(j,2)
      end if
      if (sibd(i,2).eq.sibd(j,1)) then
        sibd2(i,2)=sibd2(j,1)
      end if
      if (sibd(i,2).eq.sibd(j,2)) then
        sibd2(i,2)=sibd2(j,2)
      end if
      if (sibd2(i,1).eq.MISS) sibd2(i,1)=imiss
      return
      end
C end-of-cpibd
C
C Simulate (gene-dropping) genotypes at a single autosomal locus
C Conditioning on typed ``founder'' genotypes (true founders/marry-ins
C plus individuals without typed parents) heading informative
C chains of descent 
C
      subroutine csimped(num,nfound,fa,mo,sex,untyped,set,xlinkd)
C
C  Pedigree structure
      integer MAXSIZ, MISS
      parameter(MAXSIZ=1000, MISS=-9999)
      integer num,nfound,fa(MAXSIZ),mo(MAXSIZ),sex(MAXSIZ),set(MAXSIZ,2)
      logical untyped(MAXSIZ), xlinkd
      integer i
      logical fin
      do 15 i=nfound+1,num
      if (.not.untyped(fa(i)) .and. .not.untyped(mo(i))) then
        set(i,1)=MISS
        set(i,2)=MISS
      end if
   15 continue

C
C Main loop -- only update individuals with two typed parents
C
      iter=0
   20 continue      
        iter=iter+1
        fin=.true.
        do 30 i=nfound+1,num
        if (set(i,1).eq.MISS .and. 
     &      .not.untyped(fa(i)) .and..not.untyped(mo(i))) then
          if (set(fa(i),1).ne.MISS .and. set(mo(i),1).ne.MISS) 
     &    then
            if (xlinkd .and. sex(i).eq.1) then
              call mumson(i,mo(i),set)
            else
              call genoff(i,fa(i),mo(i),set)
            end if
          else
            fin=.false.
          end if
        end if
   30   continue
      if (.not.fin) goto 20
      return
      end
C end-of-csimped
C
C Simulate pedigree conditional on all founders (all must be typed) and 
C typed nonfounders
C
      subroutine fsimped(it,pedigree,num,nfound,id,fa,mo,set,sibd,
     &                   untyped,key,plevel)
      integer MAXSIZ, MISS 
      parameter(MAXSIZ=1000, MISS=-9999)
      integer it, plevel
C  Pedigree structure
      character*10 pedigree
      integer num, nfound
      character*10 id(MAXSIZ)
      integer fa(MAXSIZ), mo(MAXSIZ)
      integer set(MAXSIZ,2), sibd(MAXSIZ,2)
      integer key(2*MAXSIZ)
      logical untyped(MAXSIZ)
C local variables
      integer found, i, failid, maxtry, nonf, pos, restart, ord(MAXSIZ)
      logical fin

      maxtry=2000
      found=0
      do 10 i=1,nfound
        found=found+1
        sibd(i,1)=found
        key(found)=set(i,1)
        found=found+1
        sibd(i,2)=found
        key(found)=set(i,2)
   10 continue
      nonf=num-nfound
      do 12 i=1, nonf
        ord(i)=nfound+i
   12 continue
C
C start of loop -- terminated by either a successful simulation
C of ibd & genotypes of nonfounders, or bailout due <maxtry> iterations 
C without success
C
      restart=0
   40 continue
        found=0
        do 15 i=1,nfound
          if (untyped(i)) then
            key(found+1)=-set(i,1)
            key(found+2)=-set(i,2)
          end if
          found=found+2
   15   continue
        do 46 i=nfound+1,num
          sibd(i,1)=MISS
          sibd(i,2)=MISS
   46   continue

   50   continue
          fin=.true.
          call permut(nonf, ord)
          do 70 pos=1,nonf
            i=ord(pos)
          if (sibd(i,1).eq.MISS) then
            if (sibd(fa(i),1).ne.MISS .and. sibd(mo(i),1).ne.MISS) then
              call genof4(i,fa(i),mo(i),set,sibd,untyped,key,failid)
              if (failid.ne.MISS) then
                if (restart.lt.maxtry) then
                  restart=restart+1
                  goto 40
                else
                  if (plevel.gt.0) then
                    write(*,'(/a,i4,a/7x,a,a8,2a/)') 
     2 'NOTE:  In iteration ',it,' of the Metropolis algorithm, ',
     3        'simulation of ibd had to restart due to person ',
     4        id(failid),' in pedigree ',pedigree
                    if (plevel.gt.1) then
                      do 997 j=1,nfound
                        write(*,'(a10,a,2(1x,i3),a,2(1x,i3),a,l1)') 
     2                    id(j),'       x       x',set(j,1), set(j,2),
     3                    ' {', sibd(j,1),sibd(j,2), '} ',untyped(j)
  997                 continue
                      do 998 j=nfound+1,num
                        write(*,'(3a10,2(1x,i3),a,2(1x,i3),a,l1)') 
     2                    id(j),id(fa(j)),id(mo(j)), set(j,1), set(j,2),
     3                    ' {', sibd(j,1),sibd(j,2), '} ',untyped(j)
  998                 continue
                    end if
                  end if
                  call simibd(2,pedigree,num,nfound,fa,mo,set,sibd)
                  return
                end if
              end if
            else
              fin=.false.
            end if
          end if
   70     continue
      if (.not.fin) goto 50
C
      call fill2(num,set,sibd,untyped,key)
      return
      end
C end-of-fsimped
C
C  Drop ibd-alleles conditional on observed markers
C  and randomly where marker genotype not observed -- restart
C  if later generates inconsistency
C  This version assumes all founders are typed 
C 
      subroutine genof4(idx,fa,mo,set,sibd,untyped,key,failid)
      integer KNOWN, MAXSIZ, MISS
      parameter(KNOWN=0, MAXSIZ=1000, MISS=-9999)
      integer idx,fa,mo,failid
      integer set(MAXSIZ,2),sibd(MAXSIZ,2),key(2*MAXSIZ)
      logical untyped(MAXSIZ)
      integer a1,a2,first,maxtrials, second, tr1,tr2,trials
C sample without replacement from {{1,2},{1,2}}
      integer choice, i, seed, space(4)
C functions
      integer irandom

C
      failid=MISS
      maxtrials=4
      trials=0
      do 10 i=1,maxtrials
        space(i)=i
   10 continue

    1 continue

      trials=trials+1
      choice=irandom(trials, maxtrials)
      seed=space(choice)
      space(choice)=space(trials)

#if defined (F2C) || defined (SUN)
      tr1=and(seed,2)/2+1
      tr2=and(seed,1)+1
#else
      tr2=iand(seed,2)/2+1
      tr1=iand(seed,1)+1
#endif /* F2C */

      a1=sibd(fa,tr1)
      a2=sibd(mo,tr2)

C     write(*,*) 'In GENOF4():'
C      
C     write(*,*) 'Alleles: ',key(1),key(2),key(3),key(4),key(5),key(6)
C     write(*,*) 'Person #',idx,set(idx,1),'/',set(idx,2),
C    &           ' Untyped: ',untyped(idx)
C     write(*,*) 'Father #',fa, set(fa,1),'/',set(fa,2),
C    &           ' Untyped: ',untyped(fa)
C     write(*,*) 'Mother #',mo, set(mo,1),'/',set(mo,2),
C    &           ' Untyped: ',untyped(mo)
C     write(*,*) 'Transmitting: ',a1,' [',key(a1),' ], ',
C    &                            a2,' [',key(a2),' ]'
 
      if (untyped(idx)) then
        sibd(idx,1)=a1
        sibd(idx,2)=a2
      elseif (key(a1).lt.KNOWN .and. key(a2).lt.KNOWN) then
        first=irandom(1,2)
        second=3-first
        sibd(idx,first)=a1
        sibd(idx,second)=a2
        key(a1)=set(idx,first)
        key(a2)=set(idx,second)
      elseif (key(a1).lt.KNOWN .and. set(idx,1).eq.key(a2)) then
        sibd(idx,1)=a2
        sibd(idx,2)=a1
        key(a1)=set(idx,2)
      elseif (key(a1).lt.KNOWN .and. set(idx,2).eq.key(a2)) then
        sibd(idx,1)=a1
        sibd(idx,2)=a2
        key(a1)=set(idx,1)
      elseif (key(a2).lt.KNOWN .and. set(idx,2).eq.key(a1)) then
        sibd(idx,1)=a2
        sibd(idx,2)=a1
        key(a2)=set(idx,1)
      elseif (key(a2).lt.KNOWN .and. set(idx,1).eq.key(a1)) then
        sibd(idx,1)=a1
        sibd(idx,2)=a2
        key(a2)=set(idx,2)
      elseif (set(idx,1).eq.key(a1) .and. 
     &        set(idx,2).eq.key(a2)) then
        sibd(idx,1)=a1
        sibd(idx,2)=a2
      elseif (set(idx,1).eq.key(a2) .and. 
     &        set(idx,2).eq.key(a1)) then
        sibd(idx,1)=a2
        sibd(idx,2)=a1
      elseif (trials.lt.maxtrials) then
        goto 1
      else
        failid=idx
      end if
      return
      end
C end-of-genof4
C
C infer missing genotypes based on sibd values after run of fsimped
C assume all founders are typed
C
      subroutine fill2(num,set,sibd,untyped,key)
      integer MAXSIZ
      parameter(MAXSIZ=1000)
C  Pedigree structure
      integer num, set(MAXSIZ,2), sibd(MAXSIZ,2), key(2*MAXSIZ)
      logical untyped(MAXSIZ)
C local variables
      integer g1,g2,i,tmp
C
      do 5 i=1,num
      if (untyped(i)) then
        g1=abs(key(sibd(i,1)))
        g2=abs(key(sibd(i,2)))
        if (g1.gt.g2) then
          tmp=g1
          g1=g2
          g2=tmp
          tmp=sibd(i,1)
          sibd(i,1)=sibd(i,2)
          sibd(i,2)=tmp
        end if
        set(i,1)=g1
        set(i,2)=g2
      end if
    5 continue
      return
      end
C end-of-fill2
C 
C  Simulate (gene-dropping) genotypes at a single autosomal locus
C  in a pedigree of arbitrary complexity 
C
      subroutine simped(num,nfound,fa,mo,cumfrq,set)
C
C  Pedigree structure
      integer MAXSIZ, MISS
      parameter(MAXSIZ=1000, MISS=-9999)
      integer num, nfound, fa(MAXSIZ),mo(MAXSIZ), set(MAXSIZ,2)
      double precision cumfrq(*)
      integer i
      logical fin
      do 10 i=1,nfound
         call found(cumfrq,set(i,1))
         call found(cumfrq,set(i,2))
         call order(set(i,1),set(i,2))
   10 continue
      do 15 i=nfound+1,num
         set(i,1)=MISS
         set(i,2)=MISS
   15 continue

   20 continue      
        fin=.true.
        do 30 i=nfound+1,num
        if (set(i,1).eq.MISS) then
          if (set(fa(i),1).ne.MISS.and.set(mo(i),1).ne.MISS) 
     &    then
            call genoff(i,fa(i),mo(i),set)
          else
            fin=.false.
          end if
        end if
   30   continue
      if (.not.fin) goto 20
      return
      end
C end-of-simped
C 
C  Simulate (gene-dropping) genotypes at a single X-linked locus
C  in a pedigree of arbitrary complexity 
C
      subroutine xsimped(num,nfound,fa,mo,sex,cumfrq,set)
C
C  Pedigree structure
      integer MAXSIZ, MISS
      parameter(MAXSIZ=1000, MISS=-9999)
      integer num, nfound, fa(MAXSIZ),mo(MAXSIZ), sex(MAXSIZ)
      integer set(MAXSIZ,2)
      double precision cumfrq(*)
      integer i
      logical fin
C functions
      real random

      do 10 i=1,nfound
        call found(cumfrq,set(i,1))
        if (sex(i).eq.1) then
          set(i,2)=set(i,1)
        else
          call found(cumfrq,set(i,2))
          call order(set(i,1),set(i,2))
        end if
   10 continue
      do 15 i=nfound+1,num
         set(i,1)=MISS
         set(i,2)=MISS
   15 continue

   20 continue      
        fin=.true.
        do 30 i=nfound+1,num
        if (set(i,1).eq.MISS) then
          if (set(fa(i),1).ne.MISS.and.set(mo(i),1).ne.MISS) 
     &    then
            if (sex(i).eq.1) then
              if (random().gt.0.5) then
                set(i,1)=set(mo(i),1)
              else
                set(i,1)=set(mo(i),2)
              end if
              set(i,2)=set(i,1)
            else
              call genoff(i,fa(i),mo(i),set)
            end if
          else
            fin=.false.
          end if
        end if
   30   continue
      if (.not.fin) goto 20
      return
      end
C end-of-xsimped
C
C founder frequency
C
      subroutine found(cumfrq,allele)
      integer allele
      double precision cumfrq(*)
      real x
C functions
      real random
      x=random()
      allele=0
   10 continue
        allele=allele+1
      if (x.gt.cumfrq(allele)) goto 10
      return
      end
C end-of-found
C
C transmit genes from parents to child
C 
      subroutine genoff(idx,fa,mo,set)
      integer MAXSIZ
      parameter(MAXSIZ=1000)
      integer idx,fa,mo,set(MAXSIZ,2),a1,a2
      real random
      if (random().gt.0.5) then
        a1=set(fa,1)
      else
        a1=set(fa,2)
      end if
      if (random().gt.0.5) then
        a2=set(mo,1)
      else
        a2=set(mo,2)
      end if
      if (a1.gt.a2) then
        set(idx,1)=a2
        set(idx,2)=a1
      else
        set(idx,1)=a1
        set(idx,2)=a2
      end if
      return
      end
C end-of-genoff
C
C transmit single X-linked allele from mother to son
C 
      subroutine mumson(idx,mo,set)
      integer MAXSIZ
      parameter(MAXSIZ=1000)
      integer idx,mo,set(MAXSIZ,2)
      real random
      if (random().gt.0.5) then
        set(idx,1)=set(mo,1)
      else
        set(idx,1)=set(mo,2)
      end if
      set(idx,2)=set(idx,1)
      return
      end
C end-of-mumson
C 
C 
C  Given genotypes at a single locus in a pedigree of arbitrary complexity, 
C  generate ibd by gene dropping a perfectly informative marker a la
C  John Blangero.
C  Type=1 unconditional, =2, conditional on observed markers
C
      subroutine simibd(typ,pedigree,num,nfound,fa,mo,set,sibd)
C
C  Pedigree structure
      integer MAXSIZ, MISS
      parameter(MAXSIZ=1000, MISS=-9999)
      character*10 pedigree
      integer typ, num, nfound, fa(MAXSIZ),mo(MAXSIZ)
      integer set(MAXSIZ,2), sibd(MAXSIZ,2)
      integer i, ifault, found
      logical fin
      found=0
      do 10 i=1,nfound
        found=found+1
        sibd(i,1)=found
        found=found+1
        sibd(i,2)=found
   10 continue
      do 15 i=nfound+1,num
        sibd(i,1)=MISS
        sibd(i,2)=MISS
   15 continue
C
      if (typ.eq.1) then
   20   continue      
          fin=.true.
          do 30 i=nfound+1,num
          if (sibd(i,1).eq.MISS) then
            if (sibd(fa(i),1).ne.MISS.and.sibd(mo(i),1).ne.MISS) 
     &      then
              call genoff(i,fa(i),mo(i),sibd)
             else
              fin=.false.
            end if
          end if
   30     continue
        if (.not.fin) goto 20
      elseif (typ.eq.2) then
   50   continue      
          fin=.true.
          do 60 i=nfound+1,num
          if (sibd(i,1).eq.MISS) then
            if (sibd(fa(i),1).ne.MISS.and.sibd(mo(i),1).ne.MISS) 
     &      then
              ifault=0
              call genof2(pedigree,i,fa(i),mo(i),set,sibd,ifault)
              if (ifault.ne.0) then
                do 990 j=1, num
                  write(*,*) pedigree, j, fa(j), mo(j), set(j,1),
     &                      set(j,2), sibd(j,1), sibd(j,2)
  990           continue
              end if
            else
              fin=.false.
            end if
          end if
   60     continue
        if (.not.fin) goto 50
      end if
      return
      end
C end-of-simibd
C
C transmit ibd-marker from parents to child, 
C test if consistent with observed marker
C In this version 12/99, the sibd pairs are ordered by the collating
C order of the marker alleles they represent and
C not the collating order of the sibd allele.
C 
      subroutine genof2(pedigree,idx,fa,mo,set,sibd,ifault)
      integer MAXSIZ
      parameter(MAXSIZ=1000)
      character*10 pedigree
      integer idx,ifault,fa,mo,set(MAXSIZ,2),sibd(MAXSIZ,2)
      integer c1, c2, maxtrials, par1, par2, tr1, tr2, trials
C sample without replacement from {{1,2},{1,2}}
      integer choice, i, seed, space(4)
C functions
      integer irandom
C
      maxtrials=4
      trials=0
      do 10 i=1,maxtrials
        space(i)=i
   10 continue

      c1=set(idx,1)
      c2=set(idx,2)
    1 continue
        trials=trials+1
        choice=irandom(trials, maxtrials)
        seed=space(choice)
        space(choice)=space(trials)

#if defined (F2C) || defined (SUN)
        tr1=and(seed,2)/2+1
        tr2=and(seed,1)+1
#else
        tr2=iand(seed,2)/2+1
        tr1=iand(seed,1)+1
#endif /* F2C */

        par1=set(fa,tr1)
        par2=set(mo,tr2)
C
C     write(*,*) 'Prop: ',trials,' choice:',choice,
C    2   ' space: ',(space(i),i=1,4),
C    3   ' seed: ',seed,'-> ',tr1,tr2
      if (c1.eq.par1 .and. c2.eq.par2) then
        sibd(idx,1)=sibd(fa,tr1)
        sibd(idx,2)=sibd(mo,tr2)
      elseif (c1.eq.par2.and.c2.eq.par1) then
        sibd(idx,1)=sibd(mo,tr2)
        sibd(idx,2)=sibd(fa,tr1)
      elseif (trials.lt.maxtrials) then
        goto 1
      else 
        write(*,'(a/7x,a/a/)') 
     2    'ERROR: Probable mendelian inconsistency encountered',
     3    'performing ibd simulation.  Stopping prematurely.'
        write(*,*) 'In pedigree ',pedigree,', index person #',idx,
     2    ' has genotype: ',c1,c2,' Parental genotypes: ',
     3    set(fa,1),set(fa,2),' & ',set(mo,1),set(mo,2)
        ifault=1
      end if
      return
      end
C end-of-genof2
C
C count the untyped founder x founder matings for use by Metropolis algs
C
      subroutine tabmat(num,nfound,fa,mo,untyped,nummat,cntmat)
      integer MAXALL, MAXSIZ
      parameter(MAXALL=60, MAXSIZ=1000)
      integer num,nfound,fa(MAXSIZ),mo(MAXSIZ)
      logical untyped(MAXSIZ)
      integer nummat, cntmat(MAXALL,2)
C local variables
      integer i

      nummat=0
      do 5 i=1,MAXALL
        cntmat(i,1)=0
        cntmat(i,2)=0
    5 continue
      do 10 i=nfound+1,num
      if (untyped(fa(i)) .and. untyped(mo(i)) .and.
     &    fa(i).le.nfound .and. mo(i).le.nfound) then
        call mattab(fa(i),mo(i),nummat,cntmat)
      end if
   10 continue 
      return
      end
C end-of-tabmat
C
C update list of matings -- simple insertion sort
C
      subroutine mattab(key1,key2,num,cnt)
      integer MAXALL
      parameter(MAXALL=60)
      integer key1, key2, num, cnt(MAXALL,2)
      integer i,pos
      pos=0
    1 continue 
        pos=pos+1
      if (pos.le.num .and. (key1.gt.cnt(pos,1) .or.
     &    (key1.eq.cnt(pos,1) .and. key2.gt.cnt(pos,2)))) goto 1
      if (pos.gt.num .or. key2.ne.cnt(pos,2)) then
        if (num.gt.MAXALL) then
          write(*,'(a,i2,a/)') 
     &      'ERROR: More than ',MAXALL,' UnT x UnT founder matings.'
        else
          do 2 i=num,pos,-1
            cnt(i+1,1)=cnt(i,1)
            cnt(i+1,2)=cnt(i,2)
    2     continue
          num=num+1
          cnt(pos,1)=key1
          cnt(pos,2)=key2
        end if
      end if
      return
      end
C end-of-mattab
C 
C Parse a mixed model
C
      subroutine mksegmod(narg, words, trait, gt, thresh, offset,
     2                    censor, nterms, terms, nloci, loc, loctyp,
     3                    priran, nqtl, paract, par, parscal,
     4                    linkf, modtyp, shap)
      integer MAXCOL, MAXLOC, MAXPAR, MISS, RANPAR
      parameter(MAXPAR=50, MAXLOC=120, MAXCOL=MAXLOC+5, 
     &          MISS=-9999, RANPAR=24)
      integer censor, gt, linkf, modtyp, nqtl, nterms, offset, 
     &        priran, trait
      double precision shap, thresh
      integer terms(MAXLOC)
C locus names and types
      integer nloci
      character*20 loc(MAXLOC)
      integer loctyp(MAXLOC)
C command arguments
      integer narg
      character*20 words(MAXCOL)
C model pars 1=P(all) 2=a 3=d 4=AA 5=AB 6=BB 
C            7=mu 8=totvar 9=VA 10=VD 11=VG 12=VC 13=VS 14=VE 
C            15=sdG 16=sdC 17=sdS 18=sdE
C            19=a2 20=d2 21=h2 22=c2 23=s2 24=e2 
C            25..MAXPAR=Betas
      integer paract(MAXPAR)
      double precision par(MAXPAR), parscal(MAXPAR)
      character*3 keyword
      integer i, icov
      double precision p, q
C functions
      logical iscomp, isreal
      integer eow
      double precision fval

      censor=MISS
      linkf=0
      modtyp=1
      nqtl=1
      nterms=0
      offset=MISS
      priran=0
      shap=MISS 
      do 1 i=1, MAXPAR
        paract(i)=0
        par(i)=MISS 
        parscal(i)=0.0d0
    1 continue    
C default model additive QTL (free: mu, VT, a2)
      paract(7)=1
      paract(8)=1
      paract(19)=1
      paract(1)=2
      paract(2)=3
      paract(4)=3
      paract(5)=3
      paract(6)=3
      paract(9)=3
      paract(14)=3
      paract(18)=3
      paract(24)=3
      gt=0
      thresh=MISS
      i=3
      if (iscomp(words(3))) then
        call docomp(i, words, gt, thresh)
      end if

    5 if (i.gt.narg) goto 10
        if (words(i).ne.'+') keyword=words(i)(1:3)
C parameters to be estimated
        if (keyword.eq.'d' .or. keyword.eq.'vd' .or.
     &      keyword.eq.'d2') then
          paract(20)=1
          paract(3)=3
          paract(10)=3
          i=i+1
        else if (keyword.eq.'p' .or. keyword.eq.'fre') then
          paract(1)=1
          i=i+1
        else if (keyword.eq.'a' .or. keyword.eq.'va') then
          i=i+1
        else if (keyword.eq.'g' .or. keyword.eq.'vg' .or.
     &           keyword.eq.'h2') then
          paract(21)=1
          paract(11)=3
          paract(15)=3
          i=i+1
        else if (keyword.eq.'c' .or. keyword.eq.'vc' .or.
     &           keyword.eq.'c2') then
          paract(22)=1
          paract(12)=3
          paract(16)=3
          i=i+1
        else if (keyword.eq.'s' .or. keyword.eq.'vs' .or.
     &           keyword.eq.'s2') then
          paract(23)=1
          paract(13)=3
          paract(17)=3
          i=i+1
        else if (i.lt.narg .and. keyword.eq.'lin') then
          if (words(i+1).eq.'ln') then
            linkf=5
          else if (words(i+1)(1:3).eq.'mft') then
            linkf=4
          else if (words(i+1)(1:3).eq.'pro') then
            linkf=3
          else if (words(i+1)(1:3).eq.'log') then
            linkf=2
          else if (words(i+1)(1:2).eq.'id') then
            linkf=1
          else
            linkf=ival(words(i+1))
          end if
          if (linkf.eq.0) linkf=1
          i=i+2
        else if (i.lt.narg .and. keyword.eq.'lik') then
          if (words(i+1)(1:3).eq.'wei') then
            modtyp=4
          else if (words(i+1)(1:3).eq.'poi') then
            modtyp=3
          else if (words(i+1)(1:3).eq.'bin') then
            modtyp=2
          else 
            modtyp=1
          end if
          i=i+2
        else if (i.lt.narg .and. keyword.eq.'fix') then
          if (words(i+1).eq.'var') then
            paract(8)=2
            i=i+2
          else if (words(i+1).eq.'p' .or. words(i+1).eq.'fre') then
            paract(1)=2
            i=i+2
          else if (words(i+1).eq.'a' .or. words(i+1).eq.'va') then
            paract(19)=2
            i=i+2
          else if (words(i+1).eq.'d' .or. words(i+1).eq.'vd') then
            paract(20)=2
            i=i+2
          else if (words(i+1).eq.'g' .or. words(i+1).eq.'vg') then
            paract(11)=2
            paract(21)=3
            i=i+2
          else if (words(i+1).eq.'c' .or. words(i+1).eq.'vc') then
            paract(12)=2
            paract(22)=3
            i=i+2
          else if (words(i+1).eq.'s' .or. words(i+1).eq.'vs') then
            paract(13)=2
            paract(23)=3
            i=i+2
          else if (words(i+1).eq.'m' .or. words(i+1).eq.'mu') then
            paract(7)=2
            i=i+2
          else 
            write(*,'(3a)')
     &          'Cannot fix "', words(i+1)(1:eow(words(i+1))),'".'
            i=i+2
          end if
C starting values for parameters or values to evaluate likelihood at
        else if (i.lt.narg .and. isreal(words(i+1))) then
          if (keyword.eq.'ava') then
            par(2)=fval(words(i+1))
            i=i+2
          else if (keyword.eq.'dva') then
            par(3)=fval(words(i+1))
            i=i+2
          else if (keyword.eq.'pva') then
            par(1)=fval(words(i+1))
            i=i+2
          else if (keyword.eq.'mu') then
            par(7)=fval(words(i+1))
            i=i+2
          else if (keyword.eq.'var') then
            par(8)=fval(words(i+1))
            i=i+2
          else if (keyword.eq.'AA') then
            par(4)=fval(words(i+1))
            i=i+2
          else if (keyword.eq.'AB') then
            par(5)=fval(words(i+1))
            i=i+2
          else if (keyword.eq.'BB') then
            par(6)=fval(words(i+1))
            i=i+2
          else if (keyword.eq.'SD') then
            par(18)=fval(words(i+1))
            i=i+2
          else if (keyword.eq.'gva') then
            par(11)=fval(words(i+1))
            i=i+2
          else if (keyword.eq.'cva') then
            par(12)=fval(words(i+1))
            i=i+2
          else if (keyword.eq.'sva') then
            par(13)=fval(words(i+1))
            i=i+2
          else if (keyword.eq.'nqt') then
            nqtl=ival(words(i+1))
            i=i+2
          else if (keyword.eq.'sha') then
            shap=fval(words(i+1))
            i=i+2
          else 
            write(*,'(3a)') 'Skipping unknown keyword "',
     &                      words(i)(1:eow(words(i))),'".'
            i=i+1
          end if  
        else if (i.lt.narg .and. keyword.eq.'cov') then
          call gettrait(words(i+1),10,0,nloci,loc,loctyp,icov)
          if (icov.ne.MISS .and. nterms.lt.(MAXLOC-1)) then
            nterms=nterms+1
            terms(nterms)=icov
          else
            write(*,'(3a)') 'Skipping "', 
     &          words(i+1)(1:eow(words(i+1))),'": not an active trait.'
          end if
          i=i+2
        else if (i.lt.narg .and. keyword.eq.'off') then
          call gettrait(words(i+1),3,0,nloci,loc,loctyp,offset)
          if (offset.eq.MISS) then
            write(*,'(3a)') 'Variable "', words(i+1)(1:eow(words(i+1))),
     &                      '": not suitable for offset.'
          end if
          i=i+2
        else if (i.lt.narg .and. keyword.eq.'cen') then
          call gettrait(words(i+1),3,4,nloci,loc,loctyp,censor)
          if (censor.eq.MISS) then
            write(*,'(3a)') 'Variable "', words(i+1)(1:eow(words(i+1))),
     &                      '": not suitable for censoring indicator.'
          end if
          i=i+2
        else if (keyword.eq.'pri') then
          priran=1
          i=i+1
        else 
          write(*,'(3a)') 'Skipping unknown keyword "',
     &                    words(i)(1:eow(words(i))),'".'
          i=i+1
        end if
        goto 5
   10 continue
      nterms=nterms+1
      terms(nterms)=trait
C
C if nqtl=0, assume a Gaussian additive polygenic model
C
      if (nqtl.eq.0) then
        paract(1)=0
        paract(2)=0
        paract(3)=0
        paract(4)=0
        paract(5)=0
        paract(6)=0
        paract(9)=0
        paract(19)=0
        paract(20)=0
        if (paract(21).eq.0 .and. paract(22).eq.0 .and.
     &      paract(23).eq.0) then
          paract(21)=1
          paract(11)=3
          paract(15)=3
          paract(24)=3
        end if
C
C if specified p+var+genotypic means, set  a, d, mu, and then VA, VD and VE
C 1=P(all) 2=a 3=d 4=AA 5=AB 6=BB 7=mu 8=totvar 9=VA 10=VD 11=VG 12=VC 13=VS 14=VE 
C 15=sdG 16=sdC 17=sdS 18=sdE 19=a2 20=d2 21=h2 22=c2 23=s2 24=e2 25..MAXPAR=Betas
C
      else if (par(1).ne.MISS .and. par(4).ne.MISS .and. 
     2    par(5).ne.MISS .and. par(6).ne.MISS  .and.
     3    (par(8).ne. MISS .or. par(18).ne. MISS)) then
        p=par(1)
        q=1.0d0-p
        par(1)=0.5d0*abs(par(6)-par(4))
        if (paract(20).ne.0) then
          par(3)=par(5)-0.5d0*(par(4)+par(6))
        else
          par(3)=0.0d0
        end if
        par(4)=p*p*par(4) + 2*p*q*par(5) + q*q*par(6)
        par(9)=2*p*q*(par(2)+(par(1)+par(1)-1.0d0)*par(3))**2
        par(7)=4*p*q*p*q*par(3)*par(3)
        if (par(8).eq.MISS) then
          par(14)=par(18)*par(18)
          par(8)=par(9)+par(10)+par(14)
        else
          par(14)=par(8)-par(9)-par(10)
          par(8)=sqrt(par(14))
        end if
      else if (par(4).ne.MISS .or. par(5).ne.MISS .or.
     &         par(6).ne.MISS) then
        write(*,'(/2a)') 'NOTE:  Need to specify 3 genotypic means, ',
     &                   'allele frequency and variance.'
      end if
C Binomial likelihood can be explicitly requested, but is usually
C inferred using the trait type 
C If binomial or poisson likelihood, alter parameter set
      if (loctyp(trait).eq.4 .or. thresh.ne.MISS) then
        modtyp=2
      end if
      if (modtyp.eq.2 .or. modtyp.eq.3 .or. modtyp.eq.4) then
        paract(8)=0
        paract(24)=0
        paract(14)=3
        paract(18)=3
        do 100 i=21, 23
        if (paract(i).eq.1 .or. paract(i).eq.2) then
          paract(i-10)=paract(i)
          paract(i)=0
          par(i)=MISS
        end if
  100   continue
      end if
C if link function not set, set to conjugate link
      if (linkf.eq.0) then
        if (modtyp.eq.1) then
          linkf=1
        else if (modtyp.eq.2) then
          linkf=2
        else if (modtyp.eq.3 .or. modtyp.eq.4) then
          linkf=5
        end if
      end if
      return
      end
C end-of-mksegmod
C
C Names and preliminary estimates for fixed effects part of mixed model
C
      subroutine preseg(modtyp, nvar, fixed, loc, gene, numal, name, 
     &                  b, r, parnam, paract, par, parscal)
      integer MAXALL, MAXCOV, MAXIBD, MAXLOC, MAXPAR, 
     &        MAXSIZ, MAXTER, RANPAR  
      parameter(MAXALL=60, MAXLOC=120, MAXSIZ=1000, 
     2          MAXIBD=1000, MAXTER=MAXIBD/2, 
     3          MAXCOV=MAXTER*(MAXTER+1)/2, MAXPAR=50, RANPAR=24)
C
C nvar=number of variables
C gene=factor coded gene (numal-1 levels)
C nfix=number of columns in design matrix
C      
      integer gene, modtyp, numal, nvar
      integer fixed(MAXSIZ), name(MAXALL)
      character*20 loc(MAXLOC)
      character*14 parnam(MAXPAR)
      integer paract(MAXPAR)
      double precision par(MAXPAR), parscal(MAXPAR)
      double precision b(MAXTER), r(MAXCOV)
      integer i, j, ii, pos, ppos
      character*3 allel
C functions
      integer eow

      i=1
      ii=1
      par(7)=b(i)
      parscal(7)=sqrt(r(ii))
      ppos=RANPAR
      do 20 j=1, nvar-1
        pos=fixed(j)
        ncat=1 
        if (pos.eq.gene) ncat=numal-1
        do 25 k=1, ncat
          i=i+1
          ii=ii+i
          ppos=ppos+1
          par(ppos)=b(i)
          parscal(ppos)=sqrt(r(ii))
          paract(ppos)=1
          if (pos.ne.gene) then
            parnam(ppos)=loc(pos)
          else
            call wrall(name(k+1), allel)
            call juststr('l',allel,3)
            parnam(ppos)=loc(pos)(1:min(10,eow(loc(pos)))) // '*' // 
     &                   allel(1:eow(allel))
          end if
   25   continue
   20 continue
      return
      end
C end-of-preseg
C 
C Mixed model MCMC: simulate genotypes at 1..N unlinked QTLs
C                            Gaussian breeding values
C                            family intercepts
C                            regression coefficients for fixed effects
C
      subroutine segsim(wrk,wrk2,twrk,wrkfil,linkf,modtyp,shap,trait,
     2             gt, thresh, offset, censor, nfix, nvar, fixed, gene,
     3             numal, name, loc,loctyp,locpos, burnin, iter, nbatch,
     4             tune, nchain, nqtl, parnam, par, paract, parscal,
     5             mean, pedigree, num, nfound, id, fa, mo, sex, locus,
     6             numloc, set, hset, terms, cumfrq, ibdcount,mlik,
     7             mpar, priran, mcalg, plevel)
      integer IBDSIZ, MAXALL, MAXCHA, MAXHAP, MAXIBD, MAXLOC,
     &         MAXPAR, MAXPEN, MAXSIZ, MAXTER, MISS, RANPAR, SLMAX
      parameter(MAXSIZ=1000, MAXLOC=120, MISS=-9999, MAXHAP=MAXLOC/2, 
     2          MAXALL=60, MAXPAR=50, MAXCHA=4, MAXPEN=3, 
     3          MAXIBD=1000, IBDSIZ=MAXIBD*(MAXIBD+1)/2, 
     4          MAXTER=MAXIBD/2, RANPAR=24, SLMAX=10)
      integer burnin, censor, gene, gt, iter, linkf, mcalg, mpar, 
     2        modtyp, nbatch, nfix, nqtl, offset, plevel, priran, 
     3        trait, twrk, wrk, wrk2
      double precision shap, thresh
      character*144 wrkfil
      double precision mlik
C
C model pars 1=P(all) 2=a 3=d 4=AA 5=AB 6=BB 
C            7=mu 8=totvar 9=VA 10=VD 11=VG 12=VC 13=VS 14=VE 
C            15=sdG 16=sdC 17=sdS 18=sdE
C            19=a2 20=d2 21=h2 22=c2 23=s2 24=e2 
C            25..MAXPAR=Betas
C a and d are Falconer style so
C VA = 2pq(a+(p-q)d)^2
C VD = 4(pqd)^2
C tune=fudge factor for parameter proposals, usually 0.3
C linkf=link function 1=identity 2=logit 3=probit 4=MFT 5=log
C modtyp=likelihood 1=gaussian 2=binomial 3=poisson
C nchain=number of random effects chains (actually copies per family)
C nvar=number of named variables (covariates and trait)
C nfix=number of columns in design matrix
C offset=address of offset variable
C censor=address of the right censoring (binary) variable
C      
      character*14 parnam(MAXPAR)
      integer paract(MAXPAR)
      real tune
      double precision par(MAXPAR), parscal(MAXPAR)
C list of fixed effects
      integer nvar, fixed(MAXSIZ)
C and their observed means (for imputation)
      double precision mean(MAXTER)
C if a marker among the covariates, the allele names
      integer numal
      integer name(MAXALL)
C list of active random and fixed effects
      integer nterms, terms(MAXSIZ)
C Locus structure:
C number of loci, locus name, locus type, and locus position in file
      character*20 loc(MAXLOC)
      integer loctyp(MAXLOC),locpos(MAXLOC)
C Pedigree structure:
      integer actset, num, nfound, numloc
      character*10 pedigree
      character*10 id(MAXSIZ)
      integer fa(MAXSIZ), mo(MAXSIZ), sex(MAXSIZ) 
      double precision locus(MAXSIZ, MAXLOC)
C proposal and existing QTL genotypes 
      integer nchain
      integer set(MAXSIZ,2), hset(MAXSIZ,MAXHAP,2)
C
C model pars 1=P(all) 2=a 3=d 4=AA 5=AB 6=BB 
C            7=mu 8=totvar 9=VA 10=VD 11=VG 12=VC 13=VS 14=VE 
C            15=sdG 16=sdC 17=sdS 18=sdE
C            19=a2 20=d2 21=h2 22=c2 23=s2 24=e2 
C            25..MAXPAR=Betas
C
      double precision blpar(MAXPAR), bupar(MAXPAR),
     &                 epar(MAXPAR), sdpar(MAXPAR)
C model parameters
      double precision estlik, famlik, fammu, gtplik, lik,
     &                 newlik, cvlik
      double precision cumfrq(MAXALL)
      double precision glogf(MAXPEN)
      double precision val(MAXSIZ), bval(MAXSIZ), mval(MAXSIZ), 
     &                 rsd(MAXSIZ)
C work matrix to generate inbreeding for breeding values
      double precision ibdcount(IBDSIZ)
      double precision tmpval(MAXLOC)
C
C batch intercorrelations etc
C store this and previous batch to calculate order-1 autocorr
C batchit indicates which observation up to in this batch (1..sampit)
C sampit=max(1, 10*iter/nbatch), so nbatch batches per run
C curbat alternates which column is being currently updated
C proprate is acceptance rate for proposals
      integer batchit, bpar, curbat, sampit
      integer np(MAXPAR)
      integer gprop, grate, proprate(MAXPAR), proptyp(MAXPAR)
      double precision  batch(MAXPAR+3,2), covbat(MAXPAR+3),
     &                  mubat(MAXPAR+3,2), varbat(MAXPAR+3,2)
C interrupt
      integer irupt
      common /flag/ irupt
C
C Offset and censoring variables for fixed effect model are in locus()
C after the fixed effects
C
      integer cpos, nextra, opos
C
C other local variables
C maxit=10*(iter+burnin), 
C nprops=no. of proposal types, npar=no. of parameters in model
C nsubit=number of times genotype sampler run per family 
C set to tot*max(1,nqtl)/nfam
C oobounds=proposal outside bounds
C pos=current cursor pos
C ncol=nval+2 (for offset and censoring indicator)
C anyran=any random effects
C bsign=reverse sign of coefficients (eg Weibull as Poisson)
C enull=null hypothesis value for parameter (usually 0)
C likcor=1/nchains (likelihood correction for data replication)
C lmax, rmax=no. of slice expansion iterations to left and right
C
      integer anyran, bsign, i, j, maxit, ncol, nfam, nobs, npar, 
     2        nprops, nships, nsubit, nvals, oobounds, pos,
     3        tailp, tot, totobs, tpos, typ, wrknum

      double precision den, enull, likcor, sd
C
C adjust=1.0 when binomial so trait scores as 0,1
      double precision adjust
C
C baslik=offset for the likelihoods to be averaged etc; 
C ncomp=number of random effects per person;
C qtlden=multiplier to adjust QTL parameters to allow for multiple QTLS;
C hival=upper bound for pars (binomial v. continuous models); nullik=lik
C for null model proposal; pval=empirical p for null; ymean, yvar=sample
C grand mean and variance.
C
      double precision baslik, qtlden, totfrq, 
     &                 hival, nullik, pval, ymean, yvar, zstat
C empfrq=realized QTL1 allele freq in pedigrees; 
C empmu, empvar=realized pA,VA,VC,VS,VE,V(A+C+S+E)/VE in pedigrees
      integer tfound, totall
      double precision empfrq, empvar(6), empmu(6), empve, empvef
      logical last, mkstart
      character*1 ch, algid(2)
      character*8 densid(4)  
      character*8 linkid(5)
C functions
      logical complete
      integer eow, irandom
      double precision isaff, logit
C
      data densid /'Gaussian', 'Binomial', 'Poisson', 'Weibull'/
      data linkid /'Identity', 'Logit   ', 'Probit  ', 'MFT     ',
     &             'Log'/
      data algid /'S','M'/

      write(*,'(/a/a,a10,a/a)') 
     2  '------------------------------------------------',
     3  'Finite Polygenic Model analysis for "',loc(trait),'"',
     4  '------------------------------------------------' 

      npar=RANPAR+nfix
C extra parameters eg shape of multiparameter density
      if (modtyp.eq.4) then
        npar=npar+1
        parnam(npar)='log Shape'
        paract(npar)=1
      end if
C variance components estimated from distribution of individuals' random effects
      bpar=npar+1
      parnam(bpar)='Realized pA'
      paract(bpar)=0
      if (paract(1).gt.0) paract(bpar)=4
      bpar=bpar+1
      parnam(bpar)='Realized VG'
      paract(bpar)=0
      if (paract(11).gt.0 .or. par(21).gt.0) paract(bpar)=4
      bpar=bpar+1
      parnam(bpar)='Realized VC'
      paract(bpar)=0
      if (paract(12).gt.0 .or. par(22).gt.0) paract(bpar)=4
      bpar=bpar+1
      parnam(bpar)='Realized VS'
      paract(bpar)=0
      if (paract(13).gt.0 .or. par(23).gt.0) paract(bpar)=4
      bpar=bpar+1
      parnam(bpar)='Realized VE'
      paract(bpar)=4
      bpar=bpar+1
      parnam(bpar)='VE(F+R)/VE(F)'
      paract(bpar)=4

      if (nqtl .gt. MAXHAP) then
        write(*,'(a)') 'ERROR: Too many QTLs specified in fpm.'
        return
      end if
      ncol=nfix+2
      cpos=0
      nextra=0
      opos=0
      if (offset.ne.MISS) then
        nextra=nextra+1
        fixed(nvar+nextra)=offset
        opos=locpos(offset)
      end if
      if (censor.ne.MISS) then
        nextra=nextra+1
        fixed(nvar+nextra)=censor
        cpos=locpos(censor)
      end if
      wrknum=1
      tpos=locpos(trait)
C
C Initial values and bounds for model parameters
C
      bsign=1
      nterms=0
      oobounds=0

      hival=1.0d99
      adjust=0.0d0
      if (loctyp(trait).eq.4 .or. thresh.ne.MISS) adjust=1.0d0
      if (modtyp.eq.2 .and. (linkf.eq.1 .or. linkf.eq.4)) then
        hival=0.99999d0
      end if
      do 1 i=1, npar
        if (paract(i).eq.1) then
          nterms=nterms+1
          terms(nterms)=i
        end if
        np(i)=0
        proprate(i)=0
        proptyp(i)=0
    1 continue    
      do 1000 i=1, bpar
        epar(i)=0.0d0
        batch(i,1)=0.0d0
        batch(i,2)=0.0d0
        covbat(i)=0.0d0
        varbat(i,1)=0.0d0
        varbat(i,2)=0.0d0
        blpar(i)=0.00001d0 
        bupar(i)=hival  
        mubat(i,1)=0.0d0
        mubat(i,2)=0.0d0
        sdpar(i)=0.0d0
 1000 continue    
      do 1001 i=1, nfix
        blpar(RANPAR+i)=-1.0d99
        bupar(RANPAR+i)=1.0d99
 1001 continue    
C tighter bounds (0-1) on heritabilities etc 
      do 1002 i=19, 24
        bupar(i)=0.99999d0
 1002 continue    

      blpar(7)=-1.0d99
C bounds adjustments if binomial likelihood
      if (modtyp.eq.2) then
        if (linkf.eq.1) blpar(7)=0.00001d0
        bupar(8)=0.25d0
      end if
C default QTL allele frequency of 0.5
      if (par(1).eq.MISS) par(1)=0.5d0
      blpar(1)=0.000001d0
      bupar(1)=0.5d0
C Weibull model parameters need to be reversed
      if (modtyp.eq.4) then
        bsign=-1
        par(npar)=log(shap)
        parscal(npar)=0.05d0
        blpar(npar)=0.000001d0
      end if
C
      tailp=0
      baslik=0.0d0
      empfrq=0.0d0
      estlik=0.0d0
      fammu=0.0d0
      likcor=1.0d0/dfloat(nchain)
      nullik=0.0d0
      cvlik=0.0d0
      totfrq=0.0d0
      cumfrq(1)=0.5d0 
      cumfrq(2)=1.0d0
      call genfreq(par(1), cumfrq, glogf)
      qtlden=1.0d0/dfloat(max(1,nqtl))
      do 3 i=1, MAXSIZ
        bval(i)=0.0d0
        mval(i)=0.0d0
    3 continue
      gprop=0
      grate=0

      last=.false.
      nfam=0
      nships=0
      totobs=0 
      famlik=-1.0d99
      gtplik=-1.0d99
      lik=0.0d0
      ymean=0.0d0
      yvar=0.0d0
      call newnam(wrknum, wrkfil)
      open(twrk,file=wrkfil,form='unformatted')
      rewind(wrk)
    5 continue
        call wrkin(wrk,pedigree,actset,num,nfound,id,fa,mo,sex,locus,
     &             numloc, last)
        if (last) goto 20
C
        if (actset.le.0) goto 5
C
        nobs=0
        cfa=MISS
        cmo=MISS
        do 50 i=1,num
          val(i)=MISS
          if (complete(i, nvar+nextra, fixed, locpos, loctyp, locus)) 
     &    then
            nobs=nobs+1
            if (num.gt.nfound) then
              if (fa(i).ne.cfa .and. mo(i).ne.cmo) then
                cfa=fa(i)
                cmo=mo(i)
                nships=nships+1
              end if
            end if
            totobs=totobs+1
            if (thresh.ne.MISS) then
              val(i)=isaff(locus(i,tpos),thresh,gt)-adjust
            else
              val(i)=locus(i,tpos)-adjust
            end if
            call moment(totobs,val(i),ymean,yvar)
          end if
C everyone needs fixed effects -- impute missing values
          call fixeff(i, 0, gene, numal, name, nvar, fixed, 
     &                loctyp, locpos, mean, locus, tmpval)
C fixed effects model offset
          tmpval(ncol-1)=0.0d0
          if (opos.gt.0 .and. locus(i, opos).ne.MISS) then
            tmpval(ncol-1)=locus(i, opos)
          end if
C censoring variable
          tmpval(ncol)=MISS 
          if (val(i).ne.MISS) tmpval(ncol)=1.0d0
          if (cpos.gt.0 .and. locus(i, cpos).ne.MISS) then
            tmpval(ncol)=locus(i, cpos)-1.0d0
          end if
C Now transfer observation to work matrix
          do 53 j=1, ncol   
            locus(i,j)=tmpval(j)
   53     continue
   50   continue
        if (nobs.gt.0) then
          nfam=nfam+1
          call segerr(num,nfound,fa,mo,ibdcount,rsd)
          do 55 j=1, nqtl
            call simped(num,nfound,fa,mo,cumfrq,set)
            do 60 i=1, num
              hset(i,j,1)=set(i,1)
              hset(i,j,2)=set(i,2)
   60       continue
   55     continue
          do 70 i=1, nchain
            call fpmwr(twrk,ncol,nqtl,famlik,gtplik,fammu,pedigree,
     &             num,nfound,id,fa,mo,sex,locus,val,bval,rsd,mval,hset)
   70     continue
        end if
C
      goto 5
   20 continue
      yvar=yvar/dfloat(totobs)
      maxit=10*(iter+burnin)
C
C number of random effects per individual 
C Variance may be fixed 
C
      anyran=0
      ncomp=nqtl
      if (paract(11).eq.1 .or. paract(21).eq.1 .or.
     &    paract(11).eq.2 .or. paract(21).eq.2) then
        ncomp=ncomp+1
      end if
      if (paract(13).eq.1 .or. paract(23).eq.1 .or.
     &    paract(13).eq.2 .or. paract(23).eq.2) then
        ncomp=ncomp+1
      end if
      if (ncomp.gt.0 .or. paract(12).eq.1 .or. paract(22).eq.1 .or.
     &    paract(12).eq.2 .or. paract(22).eq.2) then
        anyran=1
      end if
C sampling fraction for batches
      sampit=max(1, 10*iter/nbatch)

      if (nfam.gt.0) then
        write(*,'(5(/a,i7))') 
     2    'Number of families         = ', nfam,
     3    'Number of sibships         = ', nships,
     4    'Number of observations     = ', totobs,  
     5    'Burn-in MCMC iterations    = ', 10*burnin,
     6    'Evaluated MCMC iterations  = ', 10*iter
        if (mcalg.eq.1) then
          write(*,'(a)') 'Metropolis sampler         = Sliced'
        else
          write(*,'(a)') 'Metropolis sampler         = Ordinary'
        end if
        write(*,'(2a/2a)') 
     2    'Model type                 = ', densid(modtyp),
     3    'Link type                  = ', linkid(linkf)
        if (censor.ne.MISS) then
          write(*,'(2a)') 
     2      'Censoring indicator        = ', loc(censor)
        end if
        write(*,'(3a,$)') 
     2    'Fixed Effects              = ', 
     3       loc(trait)(1:eow(loc(trait))), ' ~ mu'
        pos=34+eow(loc(trait))
        do 75 j=1, nvar-1
          write(*,'(2a,$)')
     &     ' + ', loc(fixed(j))(1:eow(loc(fixed(j))))
          pos=pos+3+eow(loc(fixed(j)))
          if (terms(j) .eq. gene) then
            write(*,'(a,$)') '(M)'
            pos=pos+3
          end if
          if (pos.gt.80) then
            pos=35
            write(*,'(/30x,a,$)') ' '
          end if
   75   continue
        if (offset.ne.MISS) then
          write(*,'(/2a)') 
     2      'Offset (for intercept)     = ', loc(offset)
        end if
        write(*,'(/a,$)') 'Random Effects             =' 
        do 80 j=9, 14
        if (paract(j).ne.0) then
          write(*,'(2a,$)') ' ', parnam(j)(1:eow(parnam(j)))
        end if
   80   continue
        write(*,'(/2(/a,f12.6))') 
     2    'Global trait mean          = ', ymean, 
     3    'Global trait variance      = ', yvar   
      else
        write(*,'(/a)') 'No eligible families!'
        close(twrk,status='delete')
        return
      end if
C
C Two alternate samplers: one genotypic, updates family by family
C                         the other, variance components, updates whole dataset
C
C Generate starting values and proposal stepsizes
      parscal(1)=0.25d0
C Total mean and variance
      if (par(7).eq.MISS) then
        par(7)=ymean
        if (modtyp.eq.2 .and. linkf.ne.1) then
          par(7)=logit(ymean)
        else if ((modtyp.eq.3 .and. linkf.eq.5) .or. modtyp.eq.4) then
          par(7)=log(ymean)
        end if
      end if
C This set by preseg() if covariates
      if (nfix.eq.0) then
        if (modtyp.eq.1) then
          parscal(7)=sqrt(yvar/dfloat(totobs))
        else if (modtyp.eq.2) then
          parscal(7)=0.05d0
        else
          parscal(7)=sqrt(par(7)/dfloat(totobs))
        end if
      end if
      if (par(8).eq.MISS) then
        par(8)=yvar
        if (linkf.eq.4) then
          par(8)=1.0d0
        end if
      end if
      parscal(8)=tune*yvar  
      if (modtyp.gt.1) parscal(8)=tune
      do 84 j=9, 14
        if (par(j).eq.MISS) then
          if (paract(j).eq.1) then
            par(j)=0.5d0
          else
            par(j)=0.0d0
          end if
        end if
        parscal(j)=sqrt(parscal(8))
   84 continue
C
C These give stepsize for random effects proposals: VG, VC, VS
C
      do 85 j=15, 18
        parscal(j)=sqrt(parscal(j-4))
   85 continue
C
      do 86 j=19, 23
        if (par(j).eq.MISS) then
          if (paract(j).eq.1) then
            par(j)=0.1d0
          else
            par(j)=0.0d0
          end if
        end if
        parscal(j)=sqrt(tune/6)
   86 continue
      call qtlmod(linkf,modtyp, par, paract, qtlden, cumfrq, glogf)
C
C Inflate number of families to total number including replicates
      nfam=nchain*nfam
C Need starting values for additive polygenic a/o sibship random
C effects?
      mkstart=(paract(11).eq.1 .or. paract(21).eq.1 .or. 
     2         paract(11).eq.2 .or. paract(21).eq.2 .or. 
     3         paract(13).eq.1 .or. paract(23).eq.1 .or.
     4         paract(13).eq.2 .or. paract(23).eq.2)
C Likelihood at starting values
      lik=0.0d0
      if (nfam.gt.1) then
        call newnam(wrknum, wrkfil)
        open(wrk2,file=wrkfil,form='unformatted')
        rewind(twrk)
        do 95 k=1, nfam
          call fpmrd(twrk,ncol,nqtl,famlik,gtplik,fammu,
     2               pedigree,num,nfound,id,fa,mo,sex,locus,
     3               val,bval,rsd,mval,hset)
          if (mkstart) then
            call initvc(par,paract,num,nfound,fa,mo,val,bval,mval)
          end if
          call seglik(linkf, modtyp, nqtl, nfix, glogf, 
     2                par,fammu, num, nfound, fa, mo, val, bval,
     3                rsd, mval, locus, hset, famlik, gtplik, plevel)
          call fpmwr(wrk2,ncol,nqtl,famlik,gtplik,fammu,
     2               pedigree,num,nfound,id,fa,mo,sex,locus,
     3               val,bval,rsd,mval,hset)
          lik=lik+famlik
   95   continue
        close(twrk,status='delete')
        close(wrk2,status='keep')
        open(twrk,file=wrkfil,form='unformatted')
      else
        if (mkstart) then
          call initvc(par,paract,num,nfound,fa,mo,val,bval,mval)
        end if
        call seglik(linkf, modtyp, nqtl, nfix, glogf, 
     2              par,fammu, num, nfound, fa, mo, val, bval,
     3              rsd, mval, locus, hset, famlik, gtplik, plevel)
        lik=lik+famlik
      end if
C
C Main loop - either propose and test a parameter tweak or a 
C   new set of QTL genotypes
C   start with a genotype proposal (typ=nterms+1)
C
      if (plevel.gt.0) then
        write(*,'(/a/2a)') 
     2    ' Parameter      Start Value  Proposal SD      Bounds',
     3    '--------------  -----------  -----------',
     4    '  --------------------'
        do 90 i=1, nterms
          write(*,'(a, 2(1x,f12.6),2(1x,d10.4))') 
     2      parnam(terms(i)), par(terms(i)), parscal(terms(i)),
     3      blpar(terms(i)), bupar(terms(i))
   90   continue
        write(*,'(/a/a)') 
     2    ' Covariate      Mean Value',
     3    '--------------  ----------' 
        do 91 i=1, nfix  
          write(*,'(a, 1x,f12.6)') parnam(RANPAR+i), mean(i)
   91   continue
        write(*,'(/a,g13.6)') 'LogLik at starting values  = ', lik 
        write(*,'(/a,20(1x,a3):)') 
     2    'MCMCPAR       it           lik  prop', 
     3    (parnam(terms(i))(1:3), i=1, nterms)
        write(*,'(a)') 
     &    'MCMCEMP       it           lik  pA VA VC VS VE  VarRat'
      end if

      it=0
      batchit=0
      curbat=1
      irupt=0
      nprops=nterms+anyran
      nvals=0
      typ=nterms+1

  100 continue
        it=it+1
        newlik=0.0d0
C update global parameter estimates via slice sampling
        if (typ.le.nterms) then
          typ=terms(typ)
          if (mcalg.eq.1) then
            call slprop(twrk, typ, linkf, modtyp, nqtl, nfix, ncol, 
     2             cumfrq, glogf, parnam, par, paract, parscal, blpar,
     3             bupar, fammu, nfam, pedigree, num, nfound, id, fa,
     4             mo, sex, val, bval, rsd, mval, locus, hset, lik,
     5             newlik, gtplik, proptyp, proprate, plevel)
          else
            call mhprop(twrk, typ, linkf, modtyp, nqtl, nfix, ncol, 
     2             cumfrq, glogf, parnam, par, paract, parscal, blpar,
     3             bupar, tune, yvar, fammu, nfam, pedigree, num,
     4             nfound, id, fa, mo, sex, val, bval, rsd, mval, locus,
     5             hset, lik, newlik, gtplik, proptyp, proprate, plevel)
          end if
          empve=0.0d0
          if (nfam.gt.1) then
            call newnam(wrknum, wrkfil)
            open(wrk2,file=wrkfil,form='unformatted')
            rewind(twrk)
            do 165 k=1, nfam
              call fpmrd(twrk,ncol,nqtl,famlik,gtplik,fammu,
     2                   pedigree,num,nfound,id,fa,mo,sex,locus,
     3                   val,bval,rsd,mval,hset)
              call seglik(linkf, modtyp, nqtl, nfix, glogf,
     2                 par, fammu, num, nfound, fa, mo, val, bval,
     3                 rsd, mval, locus, hset, famlik, gtplik, plevel)
              call fpmve(linkf, modtyp, nqtl, nfix, par, fammu,
     &                num,val, bval, mval, locus, hset, empve, empvef)
              call fpmwr(wrk2,ncol,nqtl,famlik,gtplik,fammu,
     2                   pedigree,num,nfound,id,fa,mo,sex,locus,
     3                   val,bval,rsd,mval,hset)
  165       continue
            close(twrk,status='delete')
            close(wrk2,status='keep')
            open(twrk,file=wrkfil,form='unformatted')
          else
            call fpmve(linkf, modtyp, nqtl, nfix, par, fammu,
     &             num, val, bval, mval, locus, hset, empve, empvef)
          end if
          empvar(5)=empve/dfloat(nchain*totobs)
          empvar(6)=empve/empvef
        else
C or update trait locus genotypes
          typ=0
          nships=0
          tfound=0
          tot=0
          totall=0
          empfrq=0.0d0
          empve=0.0d0
          empvef=0.0d0
          do 204 j=1, 6
            empmu(j)=0.0d0
            empvar(j)=0.0d0
  204     continue
          if (nfam.gt.1) then
            call newnam(wrknum, wrkfil)
            open(wrk2,file=wrkfil,form='unformatted')
            rewind(twrk)
            do 200 k=1, nfam
              call fpmrd(twrk,ncol,nqtl,famlik,gtplik,fammu,
     2                   pedigree,num,nfound,id,fa,mo,sex,locus,
     3                   val,bval,rsd,mval,hset)
              nsubit=max(1, num*ncomp)
              do 205 j=1, nsubit    
                call oneseg(it,linkf,modtyp,nqtl,nfix,cumfrq,
     2                 glogf,par,parscal,fammu,yvar, pedigree,num,
     3                 nfound,id,fa,mo,val,bval,rsd,mval,locus,set,
     4                 hset,famlik,gtplik,gprop,grate,mcalg,plevel)
  205         continue
              call empval(nqtl, par, num, nfound, fa, mo, hset,
     2                    fammu, bval, mval, k, tot, tfound, nships, 
     3                    empmu, empvar, totall)
              call fpmve(linkf, modtyp, nqtl, nfix, par, fammu,
     &                num, val, bval, mval, locus, hset, empve, empvef)
              call fpmwr(wrk2,ncol,nqtl,famlik,gtplik,fammu,
     2                   pedigree,num,nfound,id,fa,mo,sex,locus,
     3                   val,bval,rsd,mval,hset)
              newlik=newlik+famlik
  200       continue
            close(twrk,status='delete')
            close(wrk2,status='keep')
            open(twrk,file=wrkfil,form='unformatted')
          else
            famlik=lik
            nsubit=max(1, num*ncomp)
            do 206 j=1, nsubit
              call oneseg(it,linkf,modtyp,nqtl,nfix,cumfrq,
     2               glogf,par,parscal,fammu,yvar, pedigree,num,nfound,
     3               id,fa,mo,val,bval,rsd,mval,locus,set,hset,
     4               famlik,gtplik,gprop,grate,mcalg,plevel)
  206       continue
            call empval(nqtl, par, num, nfound, fa, mo, hset,
     2                  fammu, bval, mval, 1, tot, tfound, nships, 
     3                  empmu, empvar, totall)
            call fpmve(linkf, modtyp, nqtl, nfix, par, fammu, num,
     &             val, bval, mval, locus, hset, empve, empvef)
            newlik=famlik
          end if
          call empsum(nfam, tot, tfound, nships, empmu, empvar, 
     &                totall, empfrq)
          empvar(1)=empfrq
          empvar(5)=empve/dfloat(nchain*totobs)
          empvar(6)=empve/empvef
          if (plevel.gt.1) then
            write(*,'(a,2(a,g12.5),a,f5.3, 4(a, f9.4))') 
     2        'Shuffle genotypes',' lik=', newlik, ' (old=',lik,
     3        ') p(A)=',empfrq, ' VG=', empvar(2), ' VC=', empvar(3), 
     4        ' VS=', empvar(4), ' VE=', empvar(5)
          end if
        end if
C
C Sample from run following completion of burn-in
C Batch for standard errors
C
        if (it.gt.(10*burnin)) then
          batchit=batchit+1
          do 301 i=1, 3
            batch(i,curbat)=batch(i,curbat)+par(i)
  301     continue
C summarize genotypic means rather than genotypic deviations
          batch(4,curbat)=batch(4,curbat)+par(4)+par(7)
          batch(5,curbat)=batch(5,curbat)+par(5)+par(7)
          batch(6,curbat)=batch(6,curbat)+par(6)+par(7)
          do 304 i=7, npar
            batch(i,curbat)=batch(i,curbat)+par(i)
  304     continue
C summarize empirical random effect variances
          do 1010 i=npar+1, bpar
            batch(i,curbat)=batch(i,curbat)+empvar(i-npar)
 1010     continue
          if (batchit.eq.sampit) then
            nvals=nvals+1
            batchit=0
            do 302 i=1, bpar
              batch(i,curbat)=batch(i,curbat)/dfloat(sampit)
  302       continue
C lag-1 correlation for batches
            if (nvals.gt.1) then
              do 303 i=1, bpar
                call corr(i,batch(i,curbat),batch(i,3-curbat),
     &                    bpar,np,mubat,varbat,covbat)    
  303         continue
            end if
C
C accumulate mean and SD for parameters
C
            do 320 i=1, bpar  
              call moment(nvals, batch(i,curbat), epar(i), sdpar(i))
              batch(i,3-curbat)=0.0d0
  320       continue
            curbat=3-curbat
            if (nvals.eq.1) baslik=likcor*dfloat(int(newlik))
            call moment(nvals,likcor*newlik-baslik,estlik,cvlik)
C print BLUPs
            if (priran.eq.1 .or. plevel.gt.3) then
              if (nfam.gt.1) then
                rewind(twrk)
                do 330 k=1, nfam
                  call fpmrd(twrk,ncol,nqtl,famlik,gtplik,fammu,
     2                       pedigree,num,nfound,id,fa,mo,sex,locus,
     3                       val,bval,rsd,mval,hset)
                  call wrfpm(it, linkf, modtyp, nqtl, nfix, 
     2                   parnam, par, fammu, famlik, gtplik, pedigree,
     3                   num, nfound, id, fa, mo, sex, val, bval, mval,
     4                   locus, hset)
  330           continue
              else
                call wrfpm(it, linkf, modtyp, nqtl, nfix, parnam,
     2                 par, fammu, famlik, gtplik, pedigree, num,
     3                 nfound, id, fa, mo, sex, val, bval, mval,
     4                 locus, hset)
              end if
            end if
          end if
C Print parameter values for this iteration
          if (plevel.gt.0) then
            write(*,'(a,i7,1x,d14.8,1x,i2,(20(1x,f12.4)):)')
     2        'MCMCPAR ',it, newlik, typ, 
     3        (par(terms(i)), i=1,nterms)
            write(*,'(a,i7,1x,d14.8,1x,f5.3,5(1x,f12.4))')
     2        'MCMCEMP ',it, newlik, (empvar(j), j=1,6)
          end if
        end if
        lik=newlik
        typ=irandom(1,nprops)
      if (it.lt.maxit .and. irupt.eq.0) goto 100
C
C end of main MCMC loop
C
C model pars 1=P(all) 2=a 3=d 4=AA 5=AB 6=BB 
C            7=mu 8=totvar 9=VA 10=VD 11=VG 12=VC 13=VS 14=VE 
C            15=sdG 16=sdC 17=sdS 18=sdE
C            19=a2 20=d2 21=h2 22=c2 23=s2 24=e2 
C            25..MAXPAR=Betas
C
      estlik=estlik+baslik
      cvlik=sqrt(cvlik/dfloat(max(1, nvals-1)))
      cvlik=100.0d0*cvlik/abs(estlik)
      pval=dfloat(tailp)/dfloat(max(1,nvals))
      yvar=100.0d0
      write(*,'(/a,i5)') 
     &  'Number of simulated QTLs   = ', nqtl 
      if (nqtl.gt.0) then
        write(*,'(a,f12.6)') 
     &  'QTL increasing allele freq = ', epar(1)  
      end if
      write(*,'(/a,f12.6)')
     &  'Intercept                  = ', bsign*epar(7) 
      if (nqtl.gt.0) then
        write(*,'(a,f12.6,a,f5.1,a)')
     2  'Additive QTL variance      = ', 
     3  epar(9),' (',epar(19)*yvar,'%)'
        if (paract(10).ne.0) then
          write(*,'(a,f12.6,a,f5.1,a)')
     2   'Dominance QTL variance     = ', 
     3   epar(10),' (',epar(20)*yvar,'%)'
        end if
      end if
      if (paract(11).ne.0) then
        write(*,'(a,f12.6,a,f5.1,a)') 
     2  'Additive polygenic var     = ', 
     3  epar(11),' (',epar(21)*yvar,'%)'
      end if
      if (paract(12).ne.0) then
        write(*,'(a,f12.6,a,f5.1,a)') 
     2  'Family environmental var   = ',
     3  epar(12),' (',epar(22)*yvar,'%)'
      end if
      if (paract(13).ne.0) then
        write(*,'(a,f12.6,a,f5.1,a)') 
     2  'Maternal effect variance   = ',
     3  epar(13),' (',epar(23)*yvar,'%)'
      end if
      if (paract(14).ne.0) then
        write(*,'(a,f12.6,a,f5.1,a)')
     2    'Environmental variance     = ', 
     3    epar(14),' (',epar(24)*yvar,'%)' 
      end if
      if (modtyp.eq.4) then
        write(*,'(a,f9.3)') 
     &    'Weibull shape parameter    = ', exp(epar(RANPAR+nfix+1))
      end if
      write(*,'(a,g13.6/a,f8.2,a)')
     2  'Mean model loglikelihood   = ', estlik, 
     3  'C.V. Loglikelihood         = ', cvlik, '%'

      write(*,'(/a,i5,a,i10,a//2a/2a)') 
     2  'Based on ', nvals, ' batches of size ',sampit,' :', 
     3  'Parameter             Mean         SD    ',
     4  '    Z-value    Inter-batch r Fixed',
     5  '----------------  ----------   ----------',
     6  '   ----------  ------------- -----'
      den=sqrt(dfloat(nchain))/dfloat(max(1, nvals-1))
      call corrstd(bpar,np,varbat,covbat)
      j=0
      do 370 i=1, bpar  
      if (paract(i).ne.0) then
        bsign=1
        enull=0.0d0
C Reverse sign on Weibull parameters
        if (modtyp.eq.4) then
          if (i.eq.7 .or. (i.gt.RANPAR .and. i.le.(RANPAR+nfix))) then 
            bsign=-1
          end if
        end if
        ch=' '
        if (paract(i).eq.2) ch='*'
        sd=sqrt(den*sdpar(i))
        zstat=0.0d0
        if (sd.gt.0d0) zstat=abs(epar(i)-enull)/sd
        write(*,'(a,3(1x,f12.4),6x,f5.3,7x,a1)') 
     2    parnam(i), bsign*epar(i), sd, zstat, covbat(i), ch
        if (covbat(i).gt.0.05) j=1
      end if
  370 continue
      if (j.eq.1) then
        write(*,'(/a/7x,a)') 
     2    'NOTE:  An inter-batch correlation >0.05 suggests these ',
     3           'standard errors to be too small.'
      end if
      j=2
      if (mcalg.eq.3) j=1
      write(*,'(/a/a/a,i14,4x,f5.3,3a)') 
     2  'Proposal               N   Accepted',
     3  '------------ ------------- --------',
     4  'Genotype    ', gprop, dfloat(grate)/dfloat(max(1,gprop)),
     5  ' (',algid(j),')'
      j=mcalg
      if (mcalg.eq.3) j=1
      do 380 i=1, npar
      if (paract(i).eq.1) then
        pval=dfloat(proprate(i))/dfloat(max(1,proptyp(i)))
        write(*,'(a,i12,4x,f5.3,3a)') 
     &     parnam(i), proptyp(i), pval, ' (',algid(j),')'
      end if
  380 continue
      if (mcalg.ne.3) then
        write(*,'(/a/7x,a/7x,a)') 
     2  'NOTE:  For ordinary Metropolis sampling (M), the proposal ',
     3         'acceptance rate is optimally 0.2-0.6.  The slice ', 
     4         'sampler (S) is more robust, but slower.'  
      end if

      mpar=nterms
      mlik=-estlik-estlik
      close(twrk,status='delete')
      return
      end
C end-of-segsim
C
C QTL genotypic log frequencies
      subroutine genfreq(pall, cumfrq, glogf)
      integer MAXPEN
      parameter(MAXPEN=3)
      double precision cumfrq(*), glogf(MAXPEN), pall
      cumfrq(1)=pall
      cumfrq(2)=1.0d0
      glogf(1)=2*log(pall)
      glogf(2)=log(pall)+log(1.0d0-pall)
      glogf(3)=2*log(1.0d0-pall)
      return
      end
C end-of-genfreq
C
C dominance deviation given p, VD
      double precision function domdev(qtlden, p, vd)
      double precision p, qtlden, vd
      domdev=0.5d0*sqrt(qtlden*vd)/(p*(1.0d0-p))
      return
      end
C end-of-domev
C additive deviation given p, VA, VD
      double precision function adddev(qtlden, p, va, vd)
      double precision p, qtlden, va, vd
      double precision domdev
      adddev=(1.0d0-p-p)*domdev(qtlden, p, vd) + 
     &       sqrt(0.5d0*qtlden*va/(p*(1.0d0-p)))
      return
      end
C end-of-adddev
C
C Metropolis sampler for global parameters
C
      subroutine mhprop(twrk, iprop, linkf, modtyp, nqtl, nfix, ncol, 
     2             cumfrq, glogf, parnam, par, paract, parscal, blpar,
     3             bupar, tune, yvar, fammu, nfam, pedigree, num,
     4             nfound, id, fa, mo, sex, val, bval, rsd, mval, locus,
     5             hset, lik, newlik, gtplik, proptyp, proprate, plevel)
      implicit none
      integer MAXHAP, MAXLOC, MAXPEN, MAXSIZ, MISS, 
     &        MAXPAR, RANPAR
      parameter(MAXSIZ=1000, MAXLOC=120, MISS=-9999, 
     2          MAXHAP=MAXLOC/2, MAXPAR=50, MAXPEN=3,
     3          RANPAR=24)
      integer iprop, linkf, modtyp, ncol, nfam, nfix, nqtl, 
     &        num, nfound, plevel, twrk
      real tune
      double precision gtplik, lik, newlik, yvar
      integer paract(MAXPAR)  
      character*14 parnam(MAXPAR)
      double precision blpar(MAXPAR), bupar(MAXPAR), 
     &                 par(MAXPAR), parscal(MAXPAR)
C
C proprate is acceptance rate for proposals
C
      integer proprate(MAXPAR), proptyp(MAXPAR)
C pedigree
      character*10 pedigree
      character*10 id(MAXSIZ)
      integer fa(MAXSIZ), mo(MAXSIZ), sex(MAXSIZ)
      double precision locus(MAXSIZ, MAXLOC)
C QTL and gaussian polygenic genotypes, phenotypes
      integer hset(MAXSIZ,MAXHAP,2)
      double precision fammu
      double precision bval(MAXSIZ), rsd(MAXSIZ), val(MAXSIZ),
     &                 mval(MAXSIZ)
      double precision cumfrq(*), glogf(MAXPEN)
C local variables
      integer oobounds
      double precision origval, qtlden
C Metropolis criterion variables
      double precision lr, qa
C functions
      real random

      oobounds=0
      origval=par(iprop)
      proptyp(iprop)=proptyp(iprop)+1
      call parprop(iprop, tune, yvar, par, parscal, 
     &             blpar, bupar, oobounds)
      call pedlik(twrk, linkf, modtyp, nqtl, nfix, ncol, cumfrq, 
     2       glogf, par, paract, fammu, nfam, pedigree, num, nfound, 
     3       id, fa, mo, sex, val, bval, rsd, mval, locus, hset,
     4       newlik, gtplik, oobounds, plevel)
      if (oobounds.eq.0) then
        lr=newlik-lik
        qa=min(1.0d0,exp(lr))
      else
        qa=0.0d0
      end if
C
C If accepted, update variance components and family likelihoods
C
      if (qa.gt.random()) then
        if (plevel.gt.1) then
          write(*,'(a,2(a,1x,f9.4),2(a,g12.5),a)') 
     2      'Propose  ',parnam(iprop), origval,'->', par(iprop),
     3      ' lik=', newlik, ' (old=',lik,')  accepted '  
        end if
        proprate(iprop)=proprate(iprop)+1
      else
        if (plevel.gt.1) then
          write(*,'(a,2(a,1x,f9.4),2(a,g12.5),a)') 
     2      'Propose  ',parnam(iprop), origval,'->', par(iprop),
     3      ' lik=', newlik, ' (old=',lik,')  rejected '  
        end if
        qtlden=1.0d0/dfloat(max(1,nqtl))
        par(iprop)=origval
        call qtlmod(linkf,modtyp, par, paract, qtlden, cumfrq, glogf)
        newlik=lik
      end if
      return
      end
C end-of-mhprop
C
C Slice sampler for global parameters
C
      subroutine slprop(twrk, iprop, linkf, modtyp, nqtl, nfix, ncol, 
     2             cumfrq, glogf, parnam, par, paract, parscal, blpar,
     3             bupar, fammu, nfam, pedigree, num, nfound, id, fa,
     4             mo, sex, val, bval, rsd, mval, locus, hset, lik,
     5             newlik, gtplik, proptyp, proprate, plevel)
      integer MAXHAP, MAXLOC, MAXPEN, MAXSIZ, MISS, 
     &        MAXPAR, RANPAR, SLMAX
      parameter(MAXSIZ=1000, MAXLOC=120, MISS=-9999, 
     2          MAXHAP=MAXLOC/2, MAXPAR=50, MAXPEN=3,
     3          RANPAR=24, SLMAX=10)
      integer iprop, linkf, modtyp, ncol, nfam, nfix, nqtl, 
     &        num, nfound, plevel, twrk
      double precision gtplik, lik, newlik
      integer paract(MAXPAR)  
      character*14 parnam(MAXPAR)
      double precision blpar(MAXPAR), bupar(MAXPAR), 
     &                 par(MAXPAR), parscal(MAXPAR)
C
C proprate is acceptance rate for proposals (if sliced, proportion of
C evaluations per accepted proposal
C
      integer proprate(MAXPAR), proptyp(MAXPAR)
C pedigree
      character*10 pedigree
      character*10 id(MAXSIZ)
      integer fa(MAXSIZ), mo(MAXSIZ), sex(MAXSIZ)
      double precision locus(MAXSIZ, MAXLOC)
C QTL and gaussian polygenic genotypes, phenotypes
      integer hset(MAXSIZ,MAXHAP,2)
      double precision fammu
      double precision bval(MAXSIZ), rsd(MAXSIZ), val(MAXSIZ),
     &                 mval(MAXSIZ)
      double precision cumfrq(*), glogf(MAXPEN)
C local variables
      integer it, lmax, oobounds, rmax
C slice sampler for global parameters
      double precision ltlik, ltval, origval, rtlik, rtval, slicelik
C functions
      integer irandom
      real random

      oobounds=0
      origval=par(iprop)
C always a success, but multiple function evaluations
      proprate(iprop)=proprate(iprop)+1
      proptyp(iprop)=proptyp(iprop)+2
      slicelik=log(dble(random()))+lik
C while loops expanding the slice
      lmax=irandom(1, SLMAX)
      rmax=SLMAX-lmax
C left extent of slice
      ltval=par(iprop)-dble(random())*parscal(iprop)
      if (ltval.lt.blpar(iprop)) then
        ltval=blpar(iprop)
        lmax=0
      end if
      par(iprop)=ltval
      call parbounds(iprop, par, blpar, bupar, oobounds)
      call pedlik(twrk, linkf, modtyp, nqtl, nfix, ncol, cumfrq,
     2       glogf, par, paract, fammu, nfam, pedigree, num, nfound, 
     3       id, fa, mo, sex, val, bval, rsd, mval, locus, hset,
     4       ltlik, gtplik, oobounds, plevel)
   10 continue
      if (lmax.le.0 .or. ltlik.le.slicelik) goto 30
        ltval=ltval-parscal(iprop)
        if (ltval.lt.blpar(iprop)) then
          ltval=blpar(iprop)
          lmax=0
        end if
        par(iprop)=ltval
        call parbounds(iprop, par, blpar, bupar, oobounds)
        call pedlik(twrk, linkf, modtyp, nqtl, nfix, ncol, cumfrq, 
     2         glogf, par, paract, fammu, nfam, pedigree, num, nfound, 
     3         id, fa, mo, sex, val, bval, rsd, mval, locus, hset,
     4         ltlik, gtplik, oobounds, plevel)
        proptyp(iprop)=proptyp(iprop)+1
        lmax=lmax-1
      goto 10
   30 continue
C right extent of slice
      rtval=ltval+parscal(iprop)
      if (rtval.gt.bupar(iprop)) then
        rtval=bupar(iprop)
        rmax=0
      end if
      par(iprop)=rtval
      call parbounds(iprop, par, blpar, bupar, oobounds)
      call pedlik(twrk, linkf, modtyp, nqtl, nfix, ncol, cumfrq, glogf,
     2       par, paract, fammu, nfam, pedigree, num, nfound, 
     3       id, fa, mo, sex, val, bval, rsd, mval, locus, hset,
     4       rtlik, gtplik, oobounds, plevel)
   50 continue
      if (rmax.le.0 .or. rtlik.le.slicelik) goto 70
        rtval=rtval+parscal(iprop)
        if (rtval.gt.bupar(iprop)) then
          rtval=bupar(iprop)
          rmax=0
        end if
        par(iprop)=rtval
        call parbounds(iprop, par, blpar, bupar, oobounds)
        call pedlik(twrk, linkf, modtyp, nqtl, nfix, ncol, cumfrq, 
     2         glogf, par, paract, fammu, nfam, pedigree, num, nfound, 
     3         id, fa, mo, sex, val, bval, rsd, mval, locus, hset,
     4         rtlik, gtplik, oobounds, plevel)
        proptyp(iprop)=proptyp(iprop)+1
        rmax=rmax-1
      goto 50
   70 continue
C sample within slice, shrinking if unsuccessful
      it=0
  100 continue
        it=it+1
        proptyp(iprop)=proptyp(iprop)+1
        par(iprop)=ltval+dble(random())*(rtval-ltval)
        call parbounds(iprop, par, blpar, bupar, oobounds)
        call pedlik(twrk, linkf, modtyp, nqtl, nfix, ncol, cumfrq, 
     2         glogf, par, paract, fammu, nfam, pedigree, num, nfound, 
     3         id, fa, mo, sex, val, bval, rsd, mval, locus, hset,
     4         newlik, gtplik, oobounds, plevel)
      if (newlik.gt.slicelik .or. it.gt.(10*SLMAX)) goto 110
        if (par(iprop).lt.origval) then
          ltval=par(iprop)
          ltlik=newlik
        else
          rtval=par(iprop)
          rtlik=newlik
        end if
      goto 100 
  110 continue
      if (plevel.gt.1) then
        write(*,'(a,2(a,1x,f9.4),2(a,g12.5),a)') 
     2    'Propose  ',parnam(iprop), origval,'->', par(iprop),
     3    ' lik=', newlik, ' (old=',lik,')  accepted '  
      end if
      return
      end
C end-of-slprop
C
C Make a proposal for one QTL model parameter for Metropolis sampler
C oobounds!=0 when proposal is illegal
C
      subroutine parprop(typ, tune, yvar, par, parscal, 
     &                   blpar, bupar, oobounds)
      integer MAXPAR, RANPAR
      parameter(MAXPAR=50, RANPAR=24)
C model pars 1=P(all) 2=a 3=d 4=AA 5=AB 6=BB 
C            7=mu 8=totvar 9=VA 10=VD 11=VG 12=VC 13=VS 14=VE 
C            15=sdG 16=sdC 17=sdS 18=sdE
C            19=a2 20=d2 21=h2 22=c2 23=s2 24=e2 
C            25..MAXPAR=Betas
C
      integer oobounds, typ
      real tune
      double precision yvar
      double precision blpar(MAXPAR), bupar(MAXPAR),
     &                 par(MAXPAR), parscal(MAXPAR) 
C functions
      real randn, rantri

      if (typ.eq.7) then
C mu proposal
        par(7)=par(7)+1.5d0*dble(randn())*parscal(typ)
      else if (typ.eq.8) then
C vt proposal
        par(8)=par(8)+dble(tune*rantri())*yvar
      else if (typ.eq.1 .or. (typ.ge.19 .and. typ.le.23)) then
C p or proportion of variance due to random effect
        call triprop(tune, par(typ))
      else if (typ.le.RANPAR) then
        par(typ)=par(typ)+dble(tune*randn())*parscal(typ)
      else if (typ.gt.RANPAR) then
C fixed effect
        par(typ)=par(typ)+7.0d0*dble(tune*randn())*parscal(typ)
      end if
C bounds check
      call parbounds(typ, par, blpar, bupar, oobounds)
      return
      end
C end-of-parprop
C
C Check Metropolis proposal 
C oobounds!=0 when proposal is illegal
C
      subroutine parbounds(typ, par, blpar, bupar, oobounds)
      integer MAXPAR
      parameter(MAXPAR=50)
C model pars 1=P(all) 2=a 3=d 4=AA 5=AB 6=BB 
C            7=mu 8=totvar 9=VA 10=VD 11=VG 12=VC 13=VS 14=VE 
C            15=sdG 16=sdC 17=sdS 18=sdE
C            19=a2 20=d2 21=h2 22=c2 23=s2 24=e2 
C            25..MAXPAR=Betas
      integer oobounds, typ
      double precision blpar(MAXPAR), bupar(MAXPAR), par(MAXPAR)

      oobounds=0
      if (par(typ).lt.blpar(typ)) then
        oobounds=oobounds+1
      else if (par(typ).gt.bupar(typ)) then
        oobounds=oobounds+2
      else if ((par(19)+par(20)+par(21)+par(22)+par(23)).gt. 1.0d0) then
        oobounds=oobounds+4
      end if
      return
      end
C end-of-parbounds
C
C QTL allelic deviations, allele and genotype frequencies
C From pA, mu, totvar, a2, d2, h2, c2 calculate all other parameters
C
C model pars 1=P(all) 2=a 3=d 4=AA 5=AB 6=BB 
C            7=mu 8=totvar 9=VA 10=VD 11=VG 12=VC 13=VS 14=VE 
C            15=sdG 16=sdC 17=sdS 18=sdE
C            19=a2 20=d2 21=h2 22=c2 23=s2 24=e2 
C            25..MAXPAR=Betas
C
      subroutine qtlmod(linkf, modtyp, par, paract, qtlden, 
     &                  cumfrq, glogf)
      integer MAXPAR, MAXPEN, MISS
      parameter(MAXPAR=50, MAXPEN=3, MISS=-9999)
      integer modtyp
      integer paract(MAXPAR)
      double precision cumfrq(*), glogf(MAXPEN)
      double precision par(MAXPAR), qtlden
      double precision mu, q, var
C functions
      double precision adddev, alogit, domdev

      q=1.0d0-par(1)
      call genfreq(par(1), cumfrq, glogf)
      mu=par(7)
      var=par(8)
C SML: a2 d2
      par(9)=par(19)*var
      par(10)=par(20)*var
      par(2)=adddev(qtlden, par(1), par(9), par(10))
      par(3)=domdev(qtlden, par(1), par(10))
      par(4)=(qtlden-1.0d0) * par(7) - 
     &       2*q*(par(1)*(par(2)+par(3)) + q*par(2))
      par(5)=par(4)+par(2)+par(3)
      par(6)=par(4)+par(2)+par(2)
C  VC: h2 c2 s2
      do 10 i=11, 13
        if (paract(10+i).eq.1 .or. paract(10+i).eq.2) then
          par(i)=par(10+i)*var
        else if (paract(10+i).eq.3) then
          par(10+i)=par(i)/var
        end if
        if (par(i).gt.0.0d0) then
          par(i+4)=sqrt(par(i))
        else
          par(i+4)=1.0d-9  
        end if
   10 continue
C e2
      par(24)=1.0d0-par(19)-par(20)-par(21)-par(22)-par(23)
      if (modtyp.eq.1) then
        par(14)=par(24)*var
      else if (modtyp.eq.2) then
        if (linkf.eq.2) mu=alogit(mu)
        par(14)=mu*(1.0d0-mu)
      else if (modtyp.eq.3 .or. modtyp.eq.4) then
        if (linkf.eq.5) mu=exp(mu)
        par(14)=mu
      end if
      par(18)=sqrt(par(14))
      return
      end
C end-of-qtlmod
C
C check bounds on linear binomial model terms 
C
      subroutine chkbin(par, nqtl, oobounds)
      integer MAXPAR, MAXPEN, MISS
      parameter(MAXPAR=50, MAXPEN=3, MISS=-9999)
      integer nqtl, oobounds
      double precision par(MAXPAR)
C local variables
      integer i
      double precision pred

      do 10 i=1, 3
        pred=par(7)+nqtl*par(3+i)
        if (pred.lt.0.0d0) then
          oobounds=1
          goto 11
        else if (pred.gt.1.0d0) then
          oobounds=2
          goto 11
        end if
   10 continue
   11 continue
      return
      end
C end-of-chkbin
C 
C Circularized triangular random proportion
C
      subroutine triprop(tune, p)
      real tune
      double precision p
C functions
      real rantri
      p=p+dble(tune*rantri())
      if (p.gt.1.0d0) then
        p=p-1.0d0
      else if (p.lt.0.0d0) then
        p=1.0d0-p
      end if
      return
      end
C end-of-triprop
C
C link function for binary data
C linkf=1 identity 2 logit 3 probit 4 MFT 5=Log
C
      subroutine linfun(linkf, y, thresh)
      integer linkf
      double precision thresh, y
C functions
      double precision alogit, zp
      external alogit, zp
      if (linkf.eq.1) then
        return
      else if (linkf.eq.2) then
        y=alogit(y)
      else if (linkf.eq.3) then
        y=zp(y)
      else if (linkf.eq.4) then
        y=y-thresh
        if (y.ge.thresh) then
          y=1.0d0
        else
          y=0.0d0
        end if
      else if (linkf.eq.5) then
        y=exp(y)
      end if
      return
      end
C end-of-linfun
C
C Poisson or Binomial or Gaussian log density for FPM
      double precision function dens(x, mu, sd, shap, cens, modtyp)
      double precision SMALL
      parameter(SMALL=2.0d-200)
C modtyp: gaussian=1 binomial=1 poisson=3 weibull=4
      integer modtyp
      double precision cens, x, mu, sd, shap
      double precision dnorm, dpois, dweib, ln
      external dnorm, dpois, dweib, ln
      dens = 0.0d0
      if (modtyp.eq.1) then
        dens = dnorm(x, mu, sd)
      else if (modtyp.eq.2) then
        dens = ln(x*mu+(1.0d0-x)*(1.0d0-mu))
      else if (modtyp.eq.3) then
        if (mu.gt.SMALL) then
          dens = dpois(x, mu)
        end if
      else if (modtyp.eq.4) then
        if (mu.gt.SMALL) then
          dens = dweib(x, mu, shap, cens)
        end if
      end if
C     write(*,*) 'dens=', dens, x, mu, ' model=', modtyp
C     write(*,*) 'shap=', shap, ' cens=', cens
      return
      end
C end-of-dens 
C 
C Calculate likelihood under FPM for entire sample of pedigrees
C
      subroutine pedlik(twrk, linkf, modtyp, nqtl, nfix, ncol, 
     2                  cumfrq, glogf, par, paract, fammu, nfam,
     3                  pedigree, num, nfound, id, fa, mo, sex, val,
     4                  bval, rsd, mval, locus, hset, lik, gtplik,
     5                  oobounds, plevel)

      integer MAXHAP, MAXLOC, MAXPEN, MAXSIZ, MISS, 
     &        MAXPAR, RANPAR
      parameter(MAXSIZ=1000, MAXLOC=120, MISS=-9999, 
     2          MAXHAP=MAXLOC/2, MAXPAR=50, MAXPEN=3,
     3          RANPAR=24)
      integer linkf, modtyp, ncol, nfam, nfix, nqtl, 
     &        num, nfound, oobounds, plevel, twrk
      integer paract(MAXPAR)
      double precision par(MAXPAR)
      character*10 pedigree
      character*10 id(MAXSIZ)
      integer fa(MAXSIZ), mo(MAXSIZ), sex(MAXSIZ)
      double precision locus(MAXSIZ, MAXLOC)
C QTL and gaussian polygenic genotypes, phenotypes
      integer hset(MAXSIZ,MAXHAP,2)
      double precision fammu
      double precision bval(MAXSIZ), rsd(MAXSIZ), val(MAXSIZ),
     &                 mval(MAXSIZ)
      double precision cumfrq(*), glogf(MAXPEN)
      double precision gtplik, lik
C local variables
      integer k
      double precision famlik, qtlden

      famlik=0.0d0
      gtplik=0.0d0
      lik=0.0d0
      qtlden=1.0d0/dfloat(max(1,nqtl))
      call qtlmod(linkf, modtyp, par, paract, qtlden, cumfrq, glogf)
      if (modtyp.eq.2 .and. linkf.eq.1 .and. oobounds.eq.0) then
        call chkbin(par, nqtl, oobounds)
      end if
      if (oobounds.eq.0) then
        if (nfam.gt.1) then
          rewind(twrk)
          do 160 k=1, nfam
            call fpmrd(twrk,ncol,nqtl,famlik,gtplik,fammu,
     2                 pedigree,num,nfound,id,fa,mo,sex,locus,
     3                 val,bval,rsd,mval,hset)
            call seglik(linkf, modtyp, nqtl, nfix, glogf, 
     2              par,fammu, num, nfound, fa, mo, val, bval,
     3              rsd, mval, locus, hset, famlik, gtplik, plevel)
            lik=lik+famlik
  160     continue
        else
          call seglik(linkf, modtyp, nqtl, nfix, glogf,
     2            par, fammu, num, nfound, fa, mo, val, bval,
     3            rsd, mval, locus, hset, lik, gtplik, plevel)
        end if
      else 
        lik=-1.0d-99
      end if
      return
      end
C end-of-pedlik
C
C Calculate likelihood under FPM for one pedigree
C
      subroutine seglik(linkf, modtyp, nqtl, nfix, glogf, par,
     2                  fammu, num, nfound, fa, mo, val, bval, rsd, 
     3                  mval, locus, hset, lik, gtplik, plevel)

      integer MAXHAP, MAXLOC, MAXPEN, MAXSIZ, MISS, 
     &        MAXPAR, RANPAR
      parameter(MAXSIZ=1000, MAXLOC=120, MISS=-9999, 
     2          MAXHAP=MAXLOC/2, MAXPAR=50, MAXPEN=3,
     3          RANPAR=24)
      integer linkf, modtyp, nqtl, num, nfound, plevel
      integer fa(MAXSIZ), mo(MAXSIZ)
      double precision locus(MAXSIZ, MAXLOC)
C QTL and gaussian polygenic genotypes, phenotypes
      integer hset(MAXSIZ,MAXHAP,2)
      double precision fammu
      double precision bval(MAXSIZ), rsd(MAXSIZ), val(MAXSIZ),
     &                 mval(MAXSIZ)
C model parameters
C            1=P(all) 2=a 3=d 4=AA 5=AB 6=BB 
C            7=mu 8=totvar 9=VA 10=VD 11=VG 12=VC 13=VS 14=VE 
C            15=sdG 16=sdC 17=sdS 18=sdE
C            19=a2 20=d2 21=h2 22=c2 23=s2 24=e2 
C            25..MAXPAR=Betas
      double precision glogf(MAXPEN)
      double precision gtplik, lik, par(MAXPAR)
C local variables
      integer curmo, i
      double precision indlik, indglik, shap, yp
C functions
C     integer parcon
      double precision dens, dnorm
      external dens, dnorm

      shap=1.0d0
      if (modtyp.eq.4) shap=exp(par(RANPAR+nfix+1))
C
C evaluate likelihood for each pedigree
      gtplik=0.0d0
      lik=0.0d0
C
C Founders: different VG and VS likelihoods from those of nonfounders
      do 100 i=1,nfound
        call seglf(i, linkf, modtyp, shap, nqtl, nfix, glogf,
     2             par, fammu, num, nfound, fa, mo, val, bval, rsd, 
     3             mval, locus, hset, yp, indlik, indglik)
        lik=lik+indlik
        gtplik=gtplik+indglik
  100 continue
C
C Nonfounders
      curmo=MISS
      do 110 i=nfound+1,num
        call seglnf(i, curmo, linkf, modtyp, shap, nqtl, nfix, glogf,
     2              par, fammu, num, nfound, fa, mo, val, bval, rsd, 
     3              mval, locus, hset, yp, indlik, indglik)
        lik=lik+indlik
        gtplik=gtplik+indglik
  110 continue
C Familial environmental effect contribution
      if (par(12).gt.0.0d0) then
        gtplik=gtplik+dnorm(fammu, 0.0d0, par(16))
      end if
      lik=lik+gtplik
      return
      end
C end-of-seglik
C 
C Update (gene-dropping) genotypes at one of N unlinked QTLs
C or Gaussian polygenotype or familial random intercept
C
      subroutine oneseg(it, linkf, modtyp, nqtl, nfix, cumfrq, 
     2             glogf, par, parscal, fammu, yvar, pedigree, num, 
     3             nfound, id, fa, mo, val, bval, rsd, mval, locus, set,
     4             hset, lik, gtplik, gprop, grate, mcalg, plevel)

      integer MAXHAP, MAXLOC, MAXSIZ, MISS, MAXPAR, MAXPEN, RANPAR
      parameter(MAXSIZ=1000, MAXLOC=120, MISS=-9999, 
     &          MAXHAP=MAXLOC/2, MAXPAR=50, MAXPEN=3, RANPAR=24)
      double precision BADLIK
      parameter(BADLIK=1.0d-99)
      integer gprop, grate, it, linkf, mcalg, modtyp, nfix, nqtl,
     &         num, nfound, plevel
      double precision yvar
      character*10 pedigree
      character*10 id(MAXSIZ)
      integer fa(MAXSIZ), mo(MAXSIZ) 
      double precision locus(MAXSIZ, MAXLOC)
C proposal and existing QTL genotypes
      integer set(MAXSIZ,2), hset(MAXSIZ,MAXHAP,2)
      double precision fammu, mval(MAXSIZ)  
      double precision bval(MAXSIZ), rsd(MAXSIZ), val(MAXSIZ)
C model parameters
C  lik=model likelihood, lr=likelihood ratio newlik/lik,
C  qa=Metropolis criterion
      double precision cumfrq(*), gtplik, lik, lr, qa
      double precision glogf(MAXPEN), yp
      double precision par(MAXPAR), parscal(MAXPAR)
C local variables
      integer curmo, i, ieff, iqtl, ocurmo
      integer nspec, speceff(3)
      logical xmale
      double precision indglik, indlik, newglik, newlik, shap
      double precision newbval(MAXSIZ), newfmu, newmval(MAXSIZ)
C
      integer nchanges, changes(MAXSIZ)
C functions
      integer irandom
      real randn, random
      double precision dnorm
      external dnorm, irandom, randn, random
      
      shap=1.0d0
      if (modtyp.eq.4) shap=exp(par(RANPAR+nfix+1))
      xmale=.false.
      ieff=0
      nspec=0
      do 1 i=11, 13
      if (par(i).gt.0.0d0) then
        nspec=nspec+1
        speceff(nspec)=i
      end if
    1 continue
      if ((nqtl+nspec).eq.0) return

      iqtl=irandom(1,nqtl+nspec)
      if (plevel.gt.2) then
        if (iqtl.le.nqtl) then
          write(*,'(a,a)') '#    it pedigree   id      lik ', 
     &      'trait     pred      bval   (   sd    )     proposed'

        else
          write(*,'(a)')
     2      '#    it pedigree id lik  trait pred (sd) qtl proposed' 
        end if
      end if
      newglik=0.0d0
      newlik=0.0d0
      do 5 i=1, num
        newbval(i)=bval(i)
        newmval(i)=mval(i)
    5 continue
      newfmu=fammu
C
C evaluate likelihood for each pedigree
C   parscal 15=sdG 16=sdC 17=sdS
C
      ieff=0
      nchanges=num
      if (iqtl.gt.nqtl) then
        ieff=speceff(iqtl-nqtl)
        if (mcalg.eq.3) then
          call oneslice(ieff, linkf, modtyp, shap, nqtl, nfix, 
     2           glogf, par, parscal, fammu, pedigree, num, 
     3           nfound, id, fa, mo, val, bval, rsd, mval, locus,
     4           hset, lik, gtplik, grate, gprop, plevel)
          return
        else if (ieff.eq.11) then
          call simpol(parscal(15), num, nfound, fa, mo, newbval,
     &                nchanges, changes)
        else if (ieff.eq.12) then
          newfmu=fammu+0.5d0*parscal(16)*dble(randn())
        else
          call simmat(parscal(17), num, nfound, fa, mo, val,
     &                newmval, nchanges, changes)
        end if
      else if (irandom(1,num).eq.1) then
        call simped(num,nfound,fa,mo,cumfrq,set)
        call set2hap(iqtl, num, set, hset, 3)
      else if (iqtl.gt.0) then
        call set2hap(iqtl, num, set, hset, 2)
        call isimped(num,nfound,fa,mo,cumfrq,set)
        call set2hap(iqtl, num, set, hset, 3)
      end if
C
C Likelihood ratio if more efficient
      if (ieff.gt.0 .and. (2*nchanges).lt.num) then
        curmo=MISS
        ocurmo=MISS
        lr=0.0d0
        do 50 j=1, nchanges
          i=changes(j)
          if (i.le.nfound) then
            call seglf(i, linkf, modtyp, shap, nqtl, nfix, glogf,
     2                 par, newfmu, num, nfound, fa, mo, val, newbval, 
     3                 rsd, newmval, locus, hset, yp, indlik, indglik)
            lr=lr+indlik+indglik
            call seglf(i, linkf, modtyp, shap, nqtl, nfix, glogf,
     2                 par, fammu, num, nfound, fa, mo, val, bval, 
     3                 rsd, mval, locus, hset, yp, indlik, indglik)
            lr=lr-indlik-indglik
            if (plevel.gt.2) then
              call wronep(i,it,yp,iqtl,nqtl,ieff,indlik+indglik,par,
     2                    pedigree,id, val, bval, mval, hset, 
     3                    newbval, newmval, set)
            end if
          else
            call seglnf(i, curmo, linkf, modtyp, shap, nqtl, nfix,
     2                  glogf, par, newfmu, num, nfound, fa, mo, val,
     3                  newbval, rsd, newmval, locus, hset, yp, 
     4                  indlik, indglik)
            lr=lr+indlik+indglik
            call seglnf(i, ocurmo, linkf, modtyp, shap, nqtl, nfix,
     2                  glogf, par, fammu, num, nfound, fa, mo, val,
     3                  bval, rsd, mval, locus, hset, yp, 
     4                  indlik, indglik)
            lr=lr-indlik-indglik
            if (plevel.gt.2) then
              call wronep(i,it,yp,iqtl,nqtl,ieff,indlik+indglik,par,
     2                    pedigree,id, val, bval, mval, hset, 
     3                    newbval, newmval, set)
            end if
          end if
   50   continue
        newlik=lik+lr
      else
C
C else complete likelihood 
        do 100 i=1,nfound
          call seglf(i, linkf, modtyp, shap, nqtl, nfix, glogf,
     2           par, newfmu, num, nfound, fa, mo, val, newbval, rsd, 
     3           newmval, locus, hset, yp, indlik, indglik)
          if (plevel.gt.2) then
            call wronep(i,it,yp,iqtl,nqtl,ieff,indlik+indglik,par,
     2                  pedigree,id, val, bval, mval, hset, 
     3                  newbval, newmval, set)
          end if
          if (indlik.eq.BADLIK) then
            qa=0.0d0
            goto 115
          end if
          newlik=newlik+indlik
          newglik=newglik+indglik
  100   continue
        curmo=MISS
        do 110 i=nfound+1,num
          call seglnf(i, curmo, linkf, modtyp, shap, nqtl, nfix,glogf,
     2           par, newfmu, num, nfound, fa, mo, val, newbval, rsd,
     3           newmval, locus, hset, yp, indlik, indglik)
          if (plevel.gt.2) then
            call wronep(i,it,yp,iqtl,nqtl,ieff,indlik+indglik,par,
     2                  pedigree,id, val, bval, mval, hset, 
     3                  newbval, newmval, set)
          end if
          if (indlik.eq.BADLIK) then
            qa=0.0d0
            goto 115
          end if
          newlik=newlik+indlik
          newglik=newglik+indglik
  110   continue

C Familial environmental effect contribution
        if (par(12).gt.0.0d0) then
          newglik=newglik+dnorm(newfmu, 0.0d0, par(16))
        end if
        newlik=newlik+newglik
        lr=newlik-lik
      end if
      qa=min(1.0d0,exp(lr))
C If bounds were exceeded qa set to 0 and jumped to here:
  115 continue
C
C If accepted, update genotypes

      gprop=gprop+1
      if (qa.gt.random()) then
        grate=grate+1
        if (plevel.gt.2) then
          write(*,199) 'Proposal accepted for pedigree ', pedigree, 
     2                 ' oldgtp=', gtplik, ' newgtp=', newglik, 
     3                 ' oldlik=', lik, ' newlik=', newlik
        end if
        lik=newlik
        gtplik=newglik
        if (iqtl.gt.nqtl) then
          do 150 i=1,num
            bval(i)=newbval(i)
            mval(i)=newmval(i)
  150     continue
        end if
        fammu=newfmu
      else if (plevel.gt.2) then
        write(*,199) 'Proposal rejected for pedigree ', pedigree, 
     2               ' oldgtp=', gtplik, ' newgtp=', newglik, 
     3               ' oldlik=', lik, ' newlik=', newlik
        call set2hap(iqtl, num, set, hset, 1)
      end if
  199 format(2a,4(a,f12.4))
      return
      end
C end-of-oneseg
C 
C Slice sampler updating Gaussian polygenotype 
C
      subroutine oneslice(ieff, linkf, modtyp, shap, nqtl, nfix, 
     2             glogf, par, parscal, fammu, pedigree, num, 
     3             nfound, id, fa, mo, val, bval, rsd, mval, locus,
     4             hset, lik, gtplik, grate, gprop, plevel)

      integer MAXHAP, MAXLOC, MAXSIZ, MISS, MAXPAR, MAXPEN, RANPAR
      parameter(MAXSIZ=1000, MAXLOC=120, MISS=-9999, 
     &          MAXHAP=MAXLOC/2, MAXPAR=50, MAXPEN=3, RANPAR=24)
      double precision BADLIK
      parameter(BADLIK=1.0d-99)
      integer ieff, linkf, modtyp, nfix, nqtl, num, nfound, plevel
      double precision shap
      character*10 pedigree
      character*10 id(MAXSIZ)
      integer fa(MAXSIZ), mo(MAXSIZ) 
      double precision locus(MAXSIZ, MAXLOC)
C proposal and existing QTL genotypes
      integer hset(MAXSIZ,MAXHAP,2)
      double precision fammu, mval(MAXSIZ)  
      double precision bval(MAXSIZ), rsd(MAXSIZ), val(MAXSIZ)
C model parameters
      double precision gtplik, lik
      double precision glogf(MAXPEN)
      double precision par(MAXPAR), parscal(MAXPAR)
      integer gprop, grate
C local variables
      integer i, idx, it, lmax, rmax
      integer nchanges, changes(MAXSIZ)
      double precision baslik, newglik, newlik, ltlik, rtlik, slicelik
      double precision basgtp, ltval, origval, prop, rtval
C functions
      integer irandom
      
      it=10
C Proposal to change one random effect for this pedigree;
C change may have ramifications to offspring likelihood contribution
      nchanges=0
      if (ieff.eq.11) then
        idx=irandom(1, num)
        origval=bval(idx)
        nchanges=nchanges+1
        changes(nchanges)=idx
        do 10 i=nfound+1, num
        if (fa(i).eq.idx .or. mo(i).eq.idx) then
          nchanges=nchanges+1
          changes(nchanges)=i
        end if
   10   continue
      else if (ieff.eq.12) then
        origval=fammu
        nchanges=num
        call ascend(nchanges, changes)
      else if (ieff.eq.13) then
    5   continue
          idx=irandom(1, num)
        if (mval(idx).eq.MISS) goto 5
        origval=mval(idx)
        if (idx.gt.nfound) then
          do 15 i=nfound+1, num
          if (mo(i).eq.mo(idx)) then
            nchanges=nchanges+1
            changes(nchanges)=i
          end if
   15     continue
        else
          nchanges=nchanges+1
          changes(nchanges)=idx
        end if
      end if
C current likelihood  
      call idxlik(nchanges, changes, linkf, modtyp, shap, nqtl, nfix,
     2            glogf, par, parscal, fammu, pedigree, num, 
     3            nfound, id, fa, mo, val, bval, rsd, mval, locus,
     4            hset, baslik, basgtp)
C slice threshold and initial proposals for slice width
      slicelik=log(dble(random()))+baslik
C while loops expanding the slice
      gprop=gprop+3
      lmax=irandom(1, it)
      rmax=it-lmax
      ltval=origval-dble(random())*parscal(4+ieff)
      call sliceprop(ieff, idx, nchanges, changes, 
     &               ltval, fammu, bval, mval)
      call idxlik(nchanges, changes, linkf, modtyp, shap, nqtl, nfix,
     2            glogf, par, parscal, fammu, pedigree, num, 
     3            nfound, id, fa, mo, val, bval, rsd, mval, locus,
     4            hset, ltlik, newglik)
   20 continue
      if (lmax.eq.0 .or. ltlik.le.slicelik) goto 30
        ltval=ltval-parscal(4+ieff)
        call sliceprop(ieff, idx, nchanges, changes, 
     &                 ltval, fammu, bval, mval)
        call idxlik(nchanges, changes, linkf, modtyp, shap, nqtl, nfix,
     2              glogf, par, parscal, fammu, pedigree, num,
     3              nfound, id, fa, mo, val, bval, rsd, mval, locus,
     4              hset, ltlik, newglik)
        gprop=gprop+1
        lmax=lmax-1
      goto 20
   30 continue
C
      rtval=ltval+parscal(4+ieff)
      call sliceprop(ieff, idx, nchanges, changes, 
     &               rtval, fammu, bval, mval)
      call idxlik(nchanges, changes, linkf, modtyp, shap, nqtl, nfix,
     2            glogf, par, parscal, fammu, pedigree, num, 
     3            nfound, id, fa, mo, val, bval, rsd, mval, locus,
     4            hset, rtlik, newglik)
   40 continue
      if (rmax.eq.0 .or. rtlik.le.slicelik) goto 50
        rtval=rtval+parscal(4+ieff)
        call sliceprop(ieff, idx, nchanges, changes, 
     &                 rtval, fammu, bval, mval)
        call idxlik(nchanges, changes, linkf, modtyp, shap, nqtl, nfix,
     2              glogf, par, parscal, fammu, pedigree, num,
     3              nfound, id, fa, mo, val, bval, rsd, mval, locus,
     4              hset, rtlik, newglik)
        gprop=gprop+1
        rmax=rmax-1
      goto 40
   50 continue
C sample within slice, shrinking if unsuccessful
      it=0
   60 continue
        it=it+1
        gprop=gprop+1
        prop=ltval+dble(random())*(rtval-ltval)
        call sliceprop(ieff, idx, nchanges, changes, 
     &                 prop, fammu, bval, mval)
        call idxlik(nchanges, changes, linkf, modtyp, shap, nqtl, nfix,
     2              glogf, par, parscal, fammu, pedigree, num,
     3              nfound, id, fa, mo, val, bval, rsd, mval, locus,
     4              hset, newlik, newglik)
      if (newlik.gt.slicelik) goto 70
        if (prop.lt.origval) then
          ltval=prop
          ltlik=newlik
        else
          rtval=prop
          rtlik=newlik
        end if
      goto 60    
   70 continue
      grate=grate+1
      lik=lik+newlik-baslik
      gtplik=gtplik+newglik-basgtp
      return
      end
C end-of-oneslice
C
C copy proposal to appropriate location
C
      subroutine sliceprop(ieff, idx, nchanges, changes, 
     &                     prop, fammu, bval, mval)
      integer MAXLOC, MAXSIZ 
      parameter(MAXSIZ=1000, MAXLOC=120)
      integer idx, ieff
      integer nchanges, changes(MAXSIZ)
      double precision bval(MAXSIZ), fammu, mval(MAXSIZ), prop
      if (ieff.eq.11) then
        bval(idx)=prop 
      else if (ieff.eq.12) then
        fammu=prop 
      else if (ieff.eq.13) then
        do 10 i=1, nchanges
          mval(changes(i))=prop
   10   continue
      end if
      return
      end
C end-of-sliceprop
C 
C Gaussian polygenotype likelihood contribution of individual idx (and
C affected relatives)
C
      subroutine idxlik(nchanges, changes, linkf, modtyp, shap, nqtl,
     2             nfix, glogf, par, parscal, fammu, pedigree, 
     3             num, nfound, id, fa, mo, val, bval, rsd, mval, locus,
     4             hset, lik, gtplik)

      integer MAXHAP, MAXLOC, MAXSIZ, MISS, MAXPAR, MAXPEN, RANPAR
      parameter(MAXSIZ=1000, MAXLOC=120, MISS=-9999, 
     &          MAXHAP=MAXLOC/2, MAXPAR=50, MAXPEN=3, RANPAR=24)
      double precision BADLIK
      parameter(BADLIK=1.0d-99)
      integer linkf, modtyp, nfix, nqtl, num, nfound
      integer nchanges, changes(MAXSIZ)
      double precision shap
      character*10 pedigree
      character*10 id(MAXSIZ)
      integer fa(MAXSIZ), mo(MAXSIZ) 
      double precision locus(MAXSIZ, MAXLOC)
C proposal and existing QTL genotypes
      integer hset(MAXSIZ,MAXHAP,2)
      double precision fammu, mval(MAXSIZ)  
      double precision bval(MAXSIZ), rsd(MAXSIZ), val(MAXSIZ)
C model parameters
      double precision gtplik, lik
      double precision glogf(MAXPEN)
      double precision par(MAXPAR), parscal(MAXPAR)
C local variables
      integer curmo, idx, j
      double precision indglik, indlik, yp
C functions 
      double precision dnorm
      
      curmo=MISS
      gtplik=0.0d0
      lik=0.0d0
      do 50 j=1, nchanges
        idx=changes(j)
        if (idx.le.nfound) then
          call seglf(idx, linkf, modtyp, shap, nqtl, nfix, glogf,
     2               par, fammu, num, nfound, fa, mo, val, bval, 
     3               rsd, mval, locus, hset, yp, indlik, indglik)
        else
          call seglnf(idx, curmo, linkf, modtyp, shap, nqtl, nfix,
     2                glogf, par, fammu, num, nfound, fa, mo, val,
     3                bval, rsd, mval, locus, hset, yp, 
     4                indlik, indglik)
        end if
        if (indlik.eq.BADLIK) then
          lik=-BADLIK
          return
        end if
        gtplik=gtplik+indglik
        lik=lik+indlik
   50 continue
C Familial environmental effect contribution
      if (par(12).gt.0.0d0) then
        gtplik=gtplik+dnorm(fammu, 0.0d0, par(16))
      end if
      lik=lik+gtplik
      return
      end
C end-of-idxlik
C
C Likelihood contribution of a founder
      subroutine seglf(idx, linkf, modtyp, shap, nqtl, nfix, glogf,
     2                 par, fammu, num, nfound, fa, mo, val, bval, rsd, 
     3                 mval, locus, hset, yp, lik, gtplik)

      integer MAXHAP, MAXLOC, MAXPEN, MAXSIZ, MISS, 
     &        MAXPAR, RANPAR
      parameter(MAXSIZ=1000, MAXLOC=120, MISS=-9999, 
     2          MAXHAP=MAXLOC/2, MAXPAR=50, MAXPEN=3,
     3          RANPAR=24)
      double precision BADLIK
      parameter(BADLIK=1.0d-99)
      integer idx, linkf, modtyp, nqtl, num, nfound
      integer fa(MAXSIZ), mo(MAXSIZ)
      double precision locus(MAXSIZ, MAXLOC)
      double precision shap
C QTL and gaussian polygenic genotypes, phenotypes
      integer hset(MAXSIZ,MAXHAP,2)
      double precision fammu
      double precision bval(MAXSIZ), rsd(MAXSIZ), val(MAXSIZ),
     &                 mval(MAXSIZ)
C model parameters
C            1=P(all) 2=a 3=d 4=AA 5=AB 6=BB 
C            7=mu 8=totvar 9=VA 10=VD 11=VG 12=VC 13=VS 14=VE 
C            15=sdG 16=sdC 17=sdS 18=sdE
C            19=a2 20=d2 21=h2 22=c2 23=s2 24=e2 
C            25..MAXPAR=Betas
      double precision glogf(MAXPEN)
      double precision gtplik, lik, par(MAXPAR), yp
C local variables
      integer gtp, j
C functions
C     integer parcon
      double precision dens, dnorm
      external dens, dnorm
C
      lik=0.0d0
      gtplik=0.0d0
      yp=par(7)+bval(idx)+mval(idx)+fammu
      do 105 j=1, nfix
        yp=yp+par(RANPAR+j)*locus(idx,j)
  105 continue
C offset
      yp=yp+locus(idx, nfix+1)
C QTL contributions
      do 120 j=1, nqtl
        gtp=hset(idx,j,2)*(hset(idx,j,2)-1)/2+hset(idx,j,1)
        yp=yp+par(3+gtp)
        gtplik=gtplik+glogf(gtp)
  120 continue
      call linfun(linkf, yp, par(7))
C Bound check binary phenotypes -- why need to evaluate unphenotypeds
      if (modtyp.eq.2 .and. (yp.lt.0.0d0 .or. yp.gt.1.0d0)) then
        lik=BADLIK 
        return
      end if
C Phenotypic likelihood contribution
      if (val(idx).ne.MISS) then
        lik=lik+dens(val(idx), yp, par(18), shap, 
     &               locus(idx,nfix+2), modtyp)
C Sibship/maternal effect contribution -- 
C if founder and phenotype unobserved, does not contribute information (cf genes)
        if (par(13).gt.0.0d0) then
          gtplik=gtplik+dnorm(mval(idx), 0.0d0, par(17))
        end if
      end if
C Gaussian polygenotype contribution
      if (par(11).gt.0.0d0) then
        gtplik=gtplik+dnorm(bval(idx), 0.0d0, par(15))
      end if
      return
      end
C end-of-seglf 
C
C Likelihood contribution of a nonfounder
      subroutine seglnf(idx, curmo, linkf, modtyp, shap, nqtl, nfix, 
     2                  glogf, par, fammu, num, nfound, fa, mo, val, 
     3                  bval, rsd, mval, locus, hset, yp, lik, gtplik)

      integer MAXHAP, MAXLOC, MAXPEN, MAXSIZ, MISS, 
     &        MAXPAR, RANPAR
      parameter(MAXSIZ=1000, MAXLOC=120, MISS=-9999, 
     2          MAXHAP=MAXLOC/2, MAXPAR=50, MAXPEN=3,
     3          RANPAR=24)
      double precision BADLIK
      parameter(BADLIK=1.0d-99)
      integer curmo, idx, linkf, modtyp, nqtl, num, nfound
      integer fa(MAXSIZ), mo(MAXSIZ)
      double precision locus(MAXSIZ, MAXLOC)
C QTL and gaussian polygenic genotypes, phenotypes
      integer hset(MAXSIZ,MAXHAP,2)
      double precision fammu
      double precision bval(MAXSIZ), rsd(MAXSIZ), val(MAXSIZ),
     &                 mval(MAXSIZ)
C model parameters
C            1=P(all) 2=a 3=d 4=AA 5=AB 6=BB 
C            7=mu 8=totvar 9=VA 10=VD 11=VG 12=VC 13=VS 14=VE 
C            15=sdG 16=sdC 17=sdS 18=sdE
C            19=a2 20=d2 21=h2 22=c2 23=s2 24=e2 
C            25..MAXPAR=Betas
      double precision glogf(MAXPEN)
      double precision gtplik, lik, par(MAXPAR), shap, yp, ypf
C local variables
      double precision midpar
C functions
C     integer parcon
      double precision dens, dnorm
      external dens, dnorm
C
      lik=0.0d0
      gtplik=0.0d0
      call fpmpred(idx, modtyp, shap, nqtl, nfix, par, fammu,
     &             bval, mval, locus, hset, yp, ypf)
      call linfun(linkf, yp, par(7))
C Bound check binary phenotypes -- why need to evaluate unphenotypeds
      if (modtyp.eq.2 .and. (yp.lt.0.0d0 .or. yp.gt.1.0d0)) then
        lik=BADLIK 
        return
      end if
C Phenotypic likelihood contribution
      if (val(idx).ne.MISS) then
        lik=lik+dens(val(idx), yp, par(18), shap, 
     &               locus(idx,nfix+2), modtyp)
C Sibship/maternal effect contribution
C If no sibship members phenotyped, does not contribute to likelihood
        if (par(13).gt.0.0d0 .and. mo(idx).ne.curmo) then
          curmo=mo(idx)
          gtplik=gtplik+dnorm(mval(idx), 0.0d0, par(17))
        end if
      end if
C Gaussian polygenotype transmission contribution
      if (par(11).gt.0.0d0) then
        midpar=0.5d0*(bval(fa(idx))+bval(mo(idx)))
        gtplik=gtplik+dnorm(bval(idx), midpar, par(15)*rsd(idx))
      end if
      return
      end
C end-of-seglnf 
C 
C  Resimulate (gene-dropping) genotypes at a single autosomal locus
C  in descendants of randomly selected index 
C
      subroutine isimped(num,nfound,fa,mo,cumfrq,set)
C
C  Pedigree structure
      integer MAXSIZ, MISS
      parameter(MAXSIZ=1000, MISS=-9999)
      integer num, nfound, fa(MAXSIZ),mo(MAXSIZ), set(MAXSIZ,2)
      double precision cumfrq(*)
      integer i, idx
      logical fin
C functions
      integer irandom
C select an index person or mating and zero out descendants
      idx=irandom(1, num)
      if (idx.le.nfound .or. irandom(1,2).eq.1) then
        set(idx,1)=MISS
        set(idx,2)=MISS
      else
        set(fa(idx),1)=MISS
        set(fa(idx),2)=MISS
        set(mo(idx),1)=MISS
        set(mo(idx),2)=MISS
      end if
      do 15 i=nfound+1,num
      if (set(fa(i),1).eq.MISS .or. set(mo(i),1).eq.MISS) then
        set(i,1)=MISS
        set(i,2)=MISS
      end if
   15 continue
C fill in missing genotypes
      do 17 i=1, nfound
      if (set(i,1).eq.MISS) then
        call found(cumfrq, set(i,1))
        call found(cumfrq, set(i,2))
        call order(set(i,1),set(i,2))
      end if  
   17 continue

   20 continue      
        fin=.true.
        do 30 i=nfound+1,num
        if (set(i,1).eq.MISS) then
          if (set(fa(i),1).ne.MISS.and.set(mo(i),1).ne.MISS) 
     &    then
            call genoff(i,fa(i),mo(i),set)
          else
            fin=.false.
          end if
        end if
   30   continue
      if (.not.fin) goto 20
      return
      end
C end-of-isimped
C
C Propose updated additive polygenic values in a pedigree
C
      subroutine simpol(vsd,num,nfound,fa,mo,bval,nchanges,changes)
C
C  Pedigree structure
      integer MAXSIZ, MISS
      parameter(MAXSIZ=1000, MISS=-9999)
      integer num, nfound, fa(MAXSIZ),mo(MAXSIZ)
      integer nchanges, changes(MAXSIZ)
      double precision bval(MAXSIZ), vsd
      integer i, idx, np, sim(MAXSIZ)
C functions
      integer irandom
      real randn
C
C select an 1-4 index persons or matings or everyone
C choose to alter descendants or not
C
      do 1 i=1, num
        sim(i)=0
    1 continue
      if (irandom(1,num).eq.1) then
        do 5 i=1, num
          sim(i)=1
    5   continue
      else
        np=irandom(1,4)
        do 10 i=1, np
          idx=irandom(1, num)
          if (irandom(1,2).eq.1 .and. idx.gt.nfound) then
            sim(fa(idx))=1
            sim(mo(idx))=1
          else
            sim(idx)=1
          end if
   10   continue
        if (irandom(1,4).eq.1) then
          do 15 i=nfound+1,num
          if (sim(fa(i)).eq.1 .or. sim(mo(i)).eq.1) then
            sim(i)=1
          end if
   15     continue
        end if
      end if
C tweak indicated breeding value
      nchanges=0
      do 20 i=1, num   
      if (sim(i).eq.1) then
        nchanges=nchanges+1
        changes(nchanges)=i
        bval(i)=bval(i)+vsd*dble(randn())
      end if  
   20 continue
C also have to recalculate contribution for unaltered offspring of those with
C altered breeding values
      do 30 i=nfound+1, num   
      if (sim(i).eq.0 .and. (sim(fa(i)).eq.1 .or. sim(mo(i)).eq.1)) then
        nchanges=nchanges+1
        changes(nchanges)=i
      end if  
   30 continue
      return
      end
C end-of-simpol
C
C Propose updated sibship/maternal effects in a pedigree
C
      subroutine simmat(vstep,num,nfound,fa,mo,val,mval,
     &                  nchanges,changes)
C
C  Pedigree structure
      integer MAXSIZ, MISS
      parameter(MAXSIZ=1000, MISS=-9999)
      integer num, nfound, fa(MAXSIZ), mo(MAXSIZ),
     &        nchanges, changes(MAXSIZ) 
      double precision vstep
      double precision mval(MAXSIZ), val(MAXSIZ)
      integer elig, i, idx
C functions
      integer irandom
      real randn
C
      elig=0
      nchanges=0
      do 10 i=1, num
      if (val(i).ne.MISS) then
        elig=elig+1
        changes(elig)=i
      end if
   10 continue
C
      idx=changes(irandom(1,elig))
      mval(idx)=mval(idx)+vstep*dble(randn())
      if (idx.gt.nfound) then
        do 15 j=nfound+1, num
        if (mo(j).eq.mo(idx)) then
          mval(j)=mval(idx)
          nchanges=nchanges+1
          changes(nchanges)=j  
        end if
   15   continue
      else
        nchanges=nchanges+1
        changes(nchanges)=idx
      end if
      return
      end
C end-of-simmat
C
C Get empirical estimates of random effects
C
      subroutine empval(nqtl, par, num, nfound, fa, mo, hset,
     2                  fammu, bval, mval, ifam, tot, tfound, nships, 
     3                  empmu,  empvar, totall)
      integer MAXLOC, MAXPAR, MAXSIZ, MISS
      parameter(MAXPAR=50, MAXSIZ=1000, MAXLOC=120, 
     &          MAXHAP=MAXLOC/2, MISS=-9999)
      integer nqtl
      double precision par(MAXPAR)
      integer num, nfound, fa(MAXSIZ), mo(MAXSIZ)
      integer hset(MAXSIZ,MAXHAP,2)
      double precision fammu, bval(MAXSIZ), mval(MAXSIZ)
      integer ifam, nships, tfound, tot, totall
      double precision empmu(6), empvar(6)
C local variables
      integer i
C
C empirical allele count for 1st qtl
      if (nqtl.gt.0) then
        do 200 i=1,nfound 
          totall=totall+hset(i,1,1)+hset(i,1,2)
  200   continue
        tfound=tfound+nfound
      end if
C VG
      if (par(11).gt.0.0d0) then
        do 210 i=1, num
          tot=tot+1
          call moment(tot, bval(i), empmu(2), empvar(2))
  210   continue
      end if
C VC
      if (par(12).gt.0.0d0) then
        call moment(ifam, fammu, empmu(3), empvar(3))
      end if
C
C VS: random effect not always estimated
      if (par(13).gt.0.0d0) then
        cmo=MISS
        do 220 i=nfound+1, num
        if (mo(i).ne.cmo .and. mval(i).ne.MISS) then
          cmo=mo(i)
          nships=nships+1
          call moment(nships, mval(i), empmu(4), empvar(4))
        end if
  220   continue
      end if
      return
      end
C end-of-empval
C
C Tote up empirical statistics for random effects: VG, VC, VS
C
      subroutine empsum(nfam, tot, tfound, nships, empmu, empvar, 
     &                  totall, empfrq)
      integer nfam, nships, tfound, tot, totall
      double precision empfrq, empmu(6), empvar(6)
      if (totall.gt.0) empfrq=dfloat(4*tfound-totall)/dfloat(2*tfound)
      empvar(2)=empvar(2)/dfloat(max(1,tot-1))
      empvar(3)=empvar(3)/dfloat(max(1,nfam-1))
      empvar(4)=empvar(4)/dfloat(max(1,nships-1))
      return
      end
C end-of-empsum
C
C Initialize additive genetic and/or maternal random effects
C
      subroutine initvc(par, paract, num, nfound, fa, mo, 
     &                  val, bval, mval)
C  Pedigree structure
      integer MAXPAR, MAXSIZ, MISS
      parameter(MAXPAR=50, MAXSIZ=1000, MISS=-9999)
      integer paract(MAXPAR)
      double precision par(MAXPAR)
      integer num, nfound, fa(MAXSIZ),mo(MAXSIZ)
      double precision bval(MAXSIZ), mval(MAXSIZ), val(MAXSIZ)
      integer cfa, cmo, i
      double precision mateff, segsd
C functions
      real randn
C
      if (paract(21).eq.1 .or. paract(11).eq.1 .or.
     &    paract(21).eq.2 .or. paract(11).eq.2) then
        segsd=par(15)/sqrt(2.0d0)
        do 10 i=1, nfound
          bval(i)=dble(randn())*par(15)
   10   continue
        do 15 i=nfound+1, num
          cfa=fa(i)
          cmo=mo(i)
          midpar=0.5d0*(bval(cfa)+bval(cmo))
          bval(i)=midpar+dble(randn())*segsd
   15   continue
      end if
      if (paract(23).eq.1 .or. paract(13).eq.1 .or.
     &    paract(23).eq.2 .or. paract(13).eq.2) then
        do 50 i=1, nfound
          if (val(i).ne.MISS) then
            mval(i)=dble(randn())*par(17)
          else
            mval(i)=MISS
          end if
   50   continue
        cmo=MISS
        mateff=0.0d0
        do 60 i=nfound+1, num
          if (val(i).ne.MISS) then
            if (mo(i).ne.cmo) then
              cmo=mo(i)
              mateff=dble(randn())*par(17)
            end if
            mval(i)=mateff
          else
            mval(i)=MISS
          end if
   60   continue
      end if
      return
      end
C end-of-initvc
C 
C Copy or swap genotypes from set to hset
C 1=set->hset; 2=hset->set; 3=swap
C
      subroutine set2hap(hpos, num, set, hset, typ)
      integer MAXHAP, MAXLOC, MAXSIZ
      parameter(MAXSIZ=1000, MAXLOC=120, MAXHAP=MAXLOC/2)
      integer hpos, num, typ
C QTL genotypes
      integer hset(MAXSIZ,MAXHAP,2), set(MAXSIZ, 2)
      integer i, tmp

      if (typ.eq.1) then
        do 10 i=1, num
          hset(i,hpos,1)=set(i,1)
          hset(i,hpos,2)=set(i,2)
   10   continue
      else if (typ.eq.2) then
        do 20 i=1, num
          set(i,1)=hset(i,hpos,1)
          set(i,2)=hset(i,hpos,2)
   20   continue
      else
        do 30 i=1, num
          tmp=hset(i,hpos,1)
          hset(i,hpos,1)=set(i,1)
          set(i,1)=tmp
          tmp=hset(i,hpos,2)
          hset(i,hpos,2)=set(i,2)
          set(i,2)=tmp
   30   continue
      end if
      return
      end
C end-of-set2hap
C
C Realized VE
C
      subroutine fpmve(linkf, modtyp, nqtl, nfix, par, fammu,
     &                 num, val, bval, mval, locus, hset, ve, vef)
      integer MAXHAP, MAXLOC, MAXSIZ, MISS, MAXPAR, RANPAR
      parameter(MAXSIZ=1000, MAXLOC=120, MISS=-9999, 
     &          MAXHAP=MAXLOC/2, MAXPAR=50, RANPAR=24)
      integer linkf, modtyp, nfix, nqtl, num
      integer hset(MAXSIZ,MAXHAP,2)
      double precision locus(MAXSIZ, MAXLOC)
      double precision bval(MAXSIZ), mval(MAXSIZ), val(MAXSIZ)
      double precision fammu, par(MAXPAR), ve, vef
      integer i
      double precision shap, yp, ypf

      shap=1.0d0
      if (modtyp.eq.4) shap=exp(par(RANPAR+nfix+1))
      do 20 i=1, num
      if (val(i).ne.MISS) then
        call fpmpred(i, modtyp, shap, nqtl, nfix, par, fammu,
     &               bval, mval, locus, hset, yp, ypf)
        if (modtyp.eq.4) then
          yp=-yp/shap
          ypf=-ypf/shap
        end if
        call linfun(linkf, yp, par(7))
        call linfun(linkf, ypf, par(7))
        yp=val(i)-yp
        ve=ve+yp*yp
        ypf=val(i)-ypf
        vef=vef+ypf*ypf
      end if
   20 continue
      return
      end
C end-of-fpmve
C
C FPM Linear predictor
C
      subroutine fpmpred(idx, modtyp, shap, nqtl, nfix, par, fammu,
     &                   bval, mval, locus, hset, yp, ypf)
      integer MAXHAP, MAXLOC, MAXSIZ, MISS, MAXPAR, RANPAR
      parameter(MAXSIZ=1000, MAXLOC=120, MISS=-9999, 
     &          MAXHAP=MAXLOC/2, MAXPAR=50, RANPAR=24)
      integer idx, nfix, nqtl
      integer hset(MAXSIZ,MAXHAP,2)
      double precision locus(MAXSIZ, MAXLOC)
      double precision bval(MAXSIZ), mval(MAXSIZ)
      double precision fammu, par(MAXPAR)
C yp and ypf are linear predictor for full model and fixed effects model
C respectively
      double precision shap, yp, ypf
      integer gtp, j

      ypf=par(7)
      do 25 j=1, nfix
        ypf=ypf+par(RANPAR+j)*locus(idx,j)
   25 continue
C offset
      ypf=ypf+locus(idx, nfix+1)
      yp=ypf+fammu+bval(idx)+mval(idx)
      do 30 j=1, nqtl
        gtp=hset(idx,j,2)*(hset(idx,j,2)-1)/2+hset(idx,j,1)
        yp=yp+par(3+gtp)
   30 continue
      return
      end
C end-of-fpmpred
C
C
C Print oneseg's proposed genotypes for one individual
C
      subroutine wronep(idx, it, yp, iqtl, nqtl, ieff, lik, par, 
     2                  pedigree, id, val, bval, mval, hset, 
     3                  newbval, newmval, set)

      integer MAXHAP, MAXLOC, MAXSIZ, MISS, MAXPAR, RANPAR
      parameter(MAXSIZ=1000, MAXLOC=120, MISS=-9999, 
     &          MAXHAP=MAXLOC/2, MAXPAR=50, RANPAR=24)
      integer idx, ieff, iqtl, it, nqtl 
      character*10 pedigree
      character*10 id(MAXSIZ)
      double precision lik, yp
      integer hset(MAXSIZ,MAXHAP,2), set(MAXSIZ, 2)
      double precision bval(MAXSIZ), mval(MAXSIZ), val(MAXSIZ)
      double precision newbval(MAXSIZ), newmval(MAXSIZ)
      double precision par(MAXPAR)

      if (iqtl.le.nqtl .and. nqtl.gt.0) then
        write(*,'(i7,1x,a10,1x,a10,3(1x,f9.4),1x,3(i3,a),i3)') 
     2    it,pedigree,id(idx),lik,val(idx),yp,set(idx,1),'/',
     3    set(idx,2),' -> ',hset(idx,iqtl,1),'/',hset(idx,iqtl,2)
      else if (ieff.eq.11) then
        write(*,'(i7,1x,a10,1x,a10,4(1x,f9.4),2(a,f9.4))') 
     2    it,pedigree,id(idx),lik,val(idx),yp, 
     3    bval(idx),' (',par(15),') -> ',newbval(idx)
      else if (ieff.eq.13) then
        write(*,'(i7,1x,a10,1x,a10,4(1x,f9.4),2(a,f9.4))') 
     2    it,pedigree,id(idx),lik,val(idx),yp, 
     3    mval(idx),' (',par(17),') -> ',newmval(idx)
      end if
      return
      end
C end-of-wronep
C
C Print simulated genotypes from FPM iteration
C
      subroutine wrfpm(it, linkf, modtyp, nqtl, nfix, parnam, par,
     2             fammu, famlik, gtplik, pedigree, num, nfound, 
     3             id, fa, mo, sex, val, bval, mval, locus, hset)

      integer MAXHAP, MAXLOC, MAXSIZ, MISS, MAXPAR, RANPAR
      parameter(MAXSIZ=1000, MAXLOC=120, MISS=-9999, 
     &          MAXHAP=MAXLOC/2, MAXPAR=50, RANPAR=24)
      integer it, linkf, nfix, modtyp, nqtl, num, nfound
      character*10 pedigree
      character*10 id(MAXSIZ)
      integer fa(MAXSIZ), mo(MAXSIZ), hset(MAXSIZ,MAXHAP,2), sex(MAXSIZ)
      double precision locus(MAXSIZ, MAXLOC)
      character*14 parnam(MAXPAR)
      double precision bval(MAXSIZ), mval(MAXSIZ), val(MAXSIZ)
      double precision famlik, fammu, gtplik, par(MAXPAR)
      integer addsco, cmo, i, j, gtp
      character*1 ch
      double precision shap, sval, yp
C functions
      integer eow

      shap=1.0d0
      if (modtyp.eq.4) shap=exp(par(RANPAR+nfix+1))
      write(*,'(2a,$)') '# MCMC it pedigree id fa mo sex ',
     &   'famlik gtplik fammu trait pred bval mval qtlall'
      do 5 j=1, nqtl
        write(*,'(a,i2.2,$)') ' qtl', j
    5 continue
      do 6 j=1, nfix
        write(*,'(1x,a,$)') parnam(RANPAR+j)(1:eow(parnam(RANPAR+j)))
    6 continue
      write(*,*) 'offset censored'
C   
      cmo=MISS
      sval=0.0d0
      do 20 i=1, num
        addsco=0
C if VS>0 then unphenotyped individuals/sibships get random S value
        if (par(13).gt.0.0d0) then
          if (mval(i).ne.MISS) then
            sval=mval(i)
          else if (i.le.nfound .or. mo(i).ne.cmo) then
            cmo=mo(i)
            sval=par(17)*dble(randn())
          end if
        end if
        yp=par(7)+fammu+bval(i)+sval
        do 25 j=1, nfix
          yp=yp+par(RANPAR+j)*locus(i,j)
   25   continue
C offset
        yp=yp+locus(i, nfix+1)
        do 30 j=1, nqtl
          gtp=hset(i,j,2)*(hset(i,j,2)-1)/2+hset(i,j,1)
          addsco=addsco+gtp-1
          yp=yp+par(3+gtp)
   30   continue
        if (modtyp.eq.4) then
          yp=-yp/shap
        end if
        call linfun(linkf, yp, par(7))

        write(*,'(a,i6,2(1x,a),$)') 'MCMC ', it, pedigree, id(i)
        if (fa(i).eq.MISS) then
          write(*,'(2(1x,a10),$)') 'x         ','x         '
        else
          write(*,'(2(1x,a10),$)') id(fa(i)), id(mo(i))
        end if
        call wrsex(sex(i), ch)
        write(*,'(1x,a1,$)') ch
        write(*,'(2(1x,f12.2),f12.6,$)') famlik, gtplik, fammu
        if (val(i).eq.MISS) then
          write(*,'(6x,a,$)') 'x      '
        else
          write(*,'(1x,f12.6,$)') val(i)
        end if
        write(*,'(3(1x,f12.6),1x,i3,$)') yp, bval(i), mval(i), addsco  
        do 35 j=1, nqtl
          write(*,'(1x,i1,a1,i1)') hset(i,j,1), '/', hset(i,j,2)
   35   continue
        do 36 j=1, nfix+1
          write(*,'(1x,f9.4,$)') locus(i,j)
   36   continue
        write(*,'(1x,i1)') max(0, int(locus(i,nfix+2)))
   20 continue
      return
      end
C end-of-wrfpm
C
C Read FPM data from work file
C
      subroutine fpmrd(wrk,ncol,nqtl,famlik,gtplik,fammu,pedigree,
     2              num,nfound,id,fa,mo,sex,locus,val,bval,rsd,
     3              mval, hset)
      integer MAXSIZ,MAXLOC, MAXHAP
      parameter (MAXSIZ=1000, MAXLOC=120, MAXHAP=MAXLOC/2)
      integer ncol, nqtl, wrk
C Pedigree structure
      character*10 pedigree
      integer num, nfound
      character*10 id(MAXSIZ)
      integer fa(MAXSIZ), mo(MAXSIZ), sex(MAXSIZ)
      double precision locus(MAXSIZ,MAXLOC)
      integer hset(MAXSIZ,MAXHAP,2)
      double precision famlik, fammu, gtplik
      double precision bval(MAXSIZ), mval(MAXSIZ), rsd(MAXSIZ), 
     &                 val(MAXSIZ)  
      integer i, j
      read(wrk) pedigree, num, nfound, famlik, gtplik, fammu,
     2  (id(i), fa(i), mo(i), sex(i), 
     3   val(i), bval(i), rsd(i), mval(i), 
     4   (locus(i,j), j=1, ncol), 
     5   (hset(i,j,1), hset(i,j,2), j=1, nqtl), i=1,num)
      return
      end
C end-of-fpmrd
C
C Write FPM data to work file
C
      subroutine fpmwr(wrk,ncol,nqtl,famlik,gtplik,fammu,pedigree,
     2              num,nfound,id,fa,mo,sex,locus,val,bval,rsd,
     3              mval, hset)
      integer MAXSIZ,MAXLOC
      parameter (MAXSIZ=1000, MAXLOC=120, MAXHAP=MAXLOC/2)
      integer ncol, nqtl, wrk
C Pedigree structure
      character*10 pedigree
      integer num, nfound
      character*10 id(MAXSIZ)
      integer fa(MAXSIZ), mo(MAXSIZ), sex(MAXSIZ)
      double precision locus(MAXSIZ,MAXLOC)
      integer hset(MAXSIZ,MAXHAP,2)
      double precision famlik, fammu, gtplik
      double precision bval(MAXSIZ), mval(MAXSIZ), rsd(MAXSIZ), 
     &                 val(MAXSIZ)  
      integer i, j
      write(wrk) pedigree, num, nfound, famlik, gtplik, fammu,
     2  (id(i), fa(i), mo(i), sex(i), 
     3   val(i), bval(i), rsd(i), mval(i), 
     4   (locus(i,j), j=1, ncol), 
     5   (hset(i,j,1), hset(i,j,2), j=1, nqtl), i=1,num)
      return
      end
C end-of-fpmwr
C
C Print out simulated genotype distribution for specified persons in one
C pedigree
C
      subroutine doimp(wrk,gene,narg,words,iter,
     2             burnin,pedigree,actset,num,nfound,id,
     3             fa,mo,sex,locus,numloc,numal,name,alfrq,cumfrq,
     4             gfrq,set,set2,sibd,key,untyped,plevel)
      integer KNOWN, MAXALL, MAXCOL, MAXG, MAXSIZ,
     &        MAXLOC, MISS
      parameter(MAXALL=60, MAXG=MAXALL*(MAXALL+1)/2, MAXSIZ=1000, 
     2          MAXLOC=120, MAXCOL=MAXLOC+5, MISS=-9999, KNOWN=0) 
      integer burnin,gene,iter,plevel,wrk 
      integer narg
      character*20 words(MAXCOL)
C  Pedigree structure
      character*10 pedigree
      integer actset,num, nfound
      character*10 id(MAXSIZ)
      integer fa(MAXSIZ), mo(MAXSIZ)
      integer sex(MAXSIZ)
      double precision locus(MAXSIZ,MAXLOC)
C allele frequencies structure
      integer numal, name(MAXALL)
      double precision alfrq(MAXALL), cumfrq(MAXALL), gfrq(MAXG)
C work arrays for metropolis algorithm
      integer nuntyp
      logical untyped(MAXSIZ)
      integer set(MAXSIZ,2), sibd(MAXSIZ,2), key(2*MAXSIZ)
C metropolis work arrays
      integer set2(MAXSIZ,2)
C untyped matings
      integer nummat, cntmat(MAXALL,2)
C local variables
      integer g1, g2, gen2, i, iprop, j, nind, indx(MAXSIZ)
      logical alltyp, last, xmale
      character*10 gtp
C functions
      integer getnam

      xmale=.false.
      gen2=gene+1
      last=.false.
      rewind(wrk)
    5 continue
        call wrkin(wrk,pedigree,actset,num,nfound,id,fa,mo,sex,locus,
     &             numloc, last)
        if (last) goto 20

        if (actset.le.0 .or. (pedigree.ne.words(3))) goto 5
C
C Load selected people into indx and genotypes into set()
        write(*,*) 'Pedigree ', pedigree, ':'
        nind=0
        do 10 j=4, narg
          do 11 i=1,num
          if (words(j).eq.id(i)) then
            nind=nind+1
            indx(nind)=i
            goto 10
          end if
   11     continue
   10   continue
        do 12 i=1, nind
          call wrid('c',id(indx(i)),gtp,0)
          write(*,'(a10,$)') gtp
   12   continue
        write(*,*)

        nuntyp=0
        alltyp=.true.
        do 25 i=1,num
          if (locus(i,gene).le.KNOWN) then
            alltyp=.false.
            untyped(i)=.true.
            nuntyp=nuntyp+1
            if (locus(i,gene).eq.0 .or. locus(i,gene).eq.MISS) then
              g1=MISS
              g2=MISS
            else
              g1=getnam(-locus(i,gene),numal,name)
              g2=getnam(-locus(i,gen2),numal,name)
            end if
          else
            untyped(i)=.false.
            g1=getnam(locus(i,gene),numal,name)
            g2=getnam(locus(i,gen2),numal,name)
          end if
          call update(i,g1,g2,set)
   25   continue
        
        if (alltyp) then
          do 26 i=1, nind
            call wrgtp(name(set(indx(i),1)),name(set(indx(i),2)),gtp,1)
            write(*,'(a10,$)') gtp
   26     continue
          write(*,*) 
        else
C produce genotype frequencies for Metropolis criterion
C enumerate untyped founder matings
          call genot(numal,alfrq,ngtp,gfrq)
          call tabmat(num,nfound,fa,mo,untyped,nummat,cntmat)
C Metropolis simulation of genotypes
          if (plevel.gt.1) then
            write(*,'(/2a,4(/a,i4))') 
     2        'Metropolis simulation of pedigree ',pedigree,
     3        'Untyped Individuals: ',nuntyp,
     4        'Possible genotypes : ',ngtp,
     5        'UnT x UnT matings  : ',nummat,
     6        'Burn-in (iters)    : ',burnin
          end if
          do 44 it=1,burnin
            call drop(it,pedigree,num,nfound,id,fa,mo,nummat,
     &             cntmat,untyped,numal,gfrq,set,set2,sibd,key,iprop,0)
   44     continue
          do 45 it=1,iter
            call drop(it,pedigree,num,nfound,id,fa,mo,nummat,
     2             cntmat,untyped,numal,gfrq,set,set2,sibd,
     3             key,iprop,plevel)
            write(*,'(i2,$)') iprop
            do 46 i=1, nind
              call wrgtp(name(set(indx(i),1)),
     &                   name(set(indx(i),2)),gtp,1)
              write(*,'(a10,$)') gtp
   46       continue
          write(*,*) 
   45     continue
        end if
   20 continue
      return
      end
C end-of-doimp
C
C Monte-Carlo approach to estimating one-locus homozygosity by descent 
C
      subroutine wrhbd(wrk,twrk,gene,trait,iter,burnin,pedigree,
     2              actset,num,nfound,id,fa,mo,sex,locus, numloc,
     3              numal,name,alfrq,gfrq,untyped,set,set2,sibd,key,
     4              ibdcount, plevel)
      integer IBDSIZ, KNOWN, MAXALL, MAXG, MAXIBD, MAXSIZ, MAXLOC, MISS
      parameter(MAXALL=60, MAXG=MAXALL*(MAXALL+1)/2, MAXSIZ=1000, 
     2          MAXLOC=120, MISS=-9999, KNOWN=0, 
     3          MAXIBD=1000, IBDSIZ=MAXIBD*(MAXIBD+1)/2)
      integer burnin,gene,iter,plevel,trait,twrk,wrk 
C  Pedigree structure
      character*10 pedigree
      integer actset,num, nfound
      character*10 id(MAXSIZ)
      integer fa(MAXSIZ), mo(MAXSIZ)
      integer sex(MAXSIZ)
      double precision locus(MAXSIZ,MAXLOC)
C allele frequencies structure
      integer numal, name(MAXALL)
      double precision alfrq(MAXALL), gfrq(MAXG)
C work arrays for metropolis algorithm
      integer nuntyp
      logical untyped(MAXSIZ)
      integer set(MAXSIZ,2), sibd(MAXSIZ,2), key(2*MAXSIZ)
C metropolis work arrays
      integer set2(MAXSIZ,2)
C untyped matings
      integer nummat, cntmat(MAXALL,2)
C homozygosity-by-descent
      double precision ibdcount(MAXIBD)
C local variables
      integer g1,g2,gen2,i,idx,iprop,nfam
      logical alltyp,last,useful
      character*7 gtp
      double precision den
C functions
      integer getnam

      if (trait.eq.MISS .or. plevel.gt.1) then
        write(*,'(a)') 'Pedigree   ID        HBD    F       Gtp'
      end if

      den=1.0d0/dfloat(iter)
      gen2=gene+1
      last=.false.
      nfam=0
      rewind(wrk)
    5 continue
        call wrkin(wrk,pedigree,actset,num,nfound,id,fa,mo,sex,locus,
     &             numloc, last)
        if (last) goto 20

        if (actset.le.0) then
          if (trait.ne.MISS) then
            call wrkout(twrk,pedigree,actset,num,nfound,id,fa,mo,sex,
     &                  locus,numloc)
          end if
          goto 5
        end if
C
C test if informative pedigree -- at least one genotyped individual
C
        useful=.false.
        do 7 i=1,num
        if (locus(i,gene).gt.KNOWN) then
          useful=.true.
          goto 8
        end if
    7   continue
    8   continue
C
        if (.not.useful) then
          if (trait.ne.MISS) then
            call wrkout(twrk,pedigree,actset,num,nfound,id,fa,mo,sex,
     &                  locus,numloc)
          end if
          goto 5
        end if
C
C Load genotypes into set()
        nfam=nfam+1
        nuntyp=0
        alltyp=.true.
        do 12 i=1,num
          if (locus(i,gene).le.KNOWN) then
            alltyp=.false.
            untyped(i)=.true.
            nuntyp=nuntyp+1
            if (locus(i,gene).eq.0 .or. locus(i,gene).eq.MISS) then
              g1=MISS
              g2=MISS
            else
              g1=getnam(-locus(i,gene),numal,name)
              g2=getnam(-locus(i,gen2),numal,name)
            end if
          else
            untyped(i)=.false.
            g1=getnam(locus(i,gene),numal,name)
            g2=getnam(locus(i,gen2),numal,name)
          end if
          call update(i,g1,g2,set)
   12   continue
        call kinship(num,nfound,fa,mo,ibdcount)
        idx=0
        do 15 i=1, num
          idx=idx+i
          ibdcount(i)=ibdcount(idx)-1.0d0
   15   continue
        do 16 i=num+1, 2*num
          ibdcount(i)=0.0d0
   16   continue
C
C IBD: all genotypes known
C
        if (alltyp) then
          do 10 it=1,iter
            call simibd(2,pedigree,num,nfound,fa,mo,set,sibd)
            do 25 i=1,num
            if (sibd(i,1).eq.sibd(i,2)) then
              ibdcount(num+i)=ibdcount(num+i)+1.0d0
            end if
   25       continue
   10     continue
        else
C IBD: some untyped markers:
C produce genotype frequencies for Metropolis criterion
C enumerate untyped founder matings
          call genot(numal,alfrq,ngtp,gfrq)
          call tabmat(num,nfound,fa,mo,untyped,nummat,cntmat)
C Metropolis simulation of genotypes
          if (plevel.gt.1) then
            write(*,'(/2a,4(/a,i4))') 
     2        'Metropolis simulation of pedigree ',pedigree,
     3        'Untyped Individuals: ',nuntyp,
     4        'Possible genotypes : ',ngtp,
     5        'UnT x UnT matings  : ',nummat,
     6        'Burn-in (iters)    : ',burnin
          end if
          do 44 it=1,burnin
            call drop(it,pedigree,num,nfound,id,fa,mo,nummat,
     &             cntmat,untyped,numal,gfrq,set,set2,sibd,key,iprop,0)
   44     continue
          do 45 it=1,iter
            call drop(it,pedigree,num,nfound,id,fa,mo,nummat,
     2             cntmat,untyped,numal,gfrq,set,set2,sibd,
     3             key,iprop,plevel)
            call simibd(2,pedigree,num,nfound,fa,mo,set,sibd)
            do 65 i=1,num
            if (sibd(i,1).eq.sibd(i,2)) then
              ibdcount(num+i)=ibdcount(num+i)+1.0d0
            end if
   65       continue
   45     continue
        end if
        if (trait.eq.MISS .or. plevel.gt.1) then
          do 70 i=1,num
            if (untyped(i)) then
              call wrgtp(MISS,MISS,gtp,1)
            else
              call wrgtp(int(locus(i,gene)), 
     &                   int(locus(i,gen2)),gtp,1)
            end if
            write(*,'(a10,1x,a10,2(1x,f6.4), 1x, a7)') 
     &        pedigree, id(i), den*ibdcount(num+i), ibdcount(i), gtp
   70     continue
        end if
        if (trait.ne.MISS) then
          do 80 i=1,num
            locus(i, trait)=den*ibdcount(num+i)
   80     continue
          call wrkout(twrk,pedigree,actset,num,nfound,id,fa,mo,sex,
     &                locus,numloc)
        end if
      goto 5
   20 continue
C
      return
      end
C end-of-wrhbd  
C
C MCEM for pedigree allele frequencies
C
      subroutine mcfreq(wrk,twrk,twrk2,gene,iter,emiter,pedigree,
     3             actset,num,nfound,id,fa,mo,sex,locus,numloc,numal,
     4             name,alfrq,alfrq2,gfrq,set,set2,sibd,key,
     5             untyped,totall,typed,plevel)
      integer KNOWN, MAXALL, MAXCOL, MAXG, MAXSIZ,
     &        MAXLOC, MISS
      parameter(MAXALL=60, MAXG=MAXALL*(MAXALL+1)/2, MAXSIZ=1000, 
     2          MAXLOC=120, MAXCOL=MAXLOC+5, MISS=-9999, KNOWN=0) 
      integer emiter, gene, iter, plevel, totall,twrk,twrk2,typed,wrk 
C  Pedigree structure
      character*10 pedigree
      integer actset,num, nfound
      character*10 id(MAXSIZ)
      integer fa(MAXSIZ), mo(MAXSIZ)
      integer sex(MAXSIZ)
      double precision locus(MAXSIZ,MAXLOC)
C allele frequencies structure
      integer numal, name(MAXALL)
      double precision alfrq(MAXALL), alfrq2(MAXALL), 
     &                 gfrq(MAXG)
C work arrays for metropolis algorithm
      integer nuntyp
      logical untyped(MAXSIZ)
      integer set(MAXSIZ,2), sibd(MAXSIZ,2), key(2*MAXSIZ)
C metropolis work arrays
      integer set2(MAXSIZ,2)
C interrupt
      integer irupt
      common /flag/ irupt
C untyped matings
      integer nummat, cntmat(MAXALL,2)
C local variables
      integer emit, g1, g2, gen2, i, iprop, j, maxem, nfam, 
     &        ntyped, tfound, wrknum
      integer proprate(4), proptyp(4)
      character*12 wrkfil
      logical alltyp,last
      double precision accel, delta, den, one,  wei
C functions
      integer getnam  

      if (numal.eq.0) then
        return
      end if

      wrknum=1
      wrkfil='sp-mcf.wrk'
      call newnam(wrknum, wrkfil)
      open(twrk,file=wrkfil,form='unformatted')

      gen2=gene+1
      nfam=0
      ntyped=0
      nuntyp=0
      tfound=0
      one=1.0d0
      wei=1.0d0/dfloat(iter)
      last=.false.
      rewind(wrk)
    5 continue
        call wrkin(wrk,pedigree,actset,num,nfound,id,fa,mo,sex,locus,
     &             numloc, last)
        if (last) goto 20

        if (actset.le.0) goto 5

        nfam=nfam+1
        tfound=tfound+nfound
        alltyp=.true.
        do 25 i=1,num
          if (locus(i,gene).le.KNOWN) then
            if (i.le.nfound) then
              alltyp=.false.
              nuntyp=nuntyp+1
            end if
            untyped(i)=.true.
            if (locus(i,gene).eq.0 .or. locus(i,gene).eq.MISS) then
              g1=MISS
              g2=MISS
            else
              g1=getnam(-locus(i,gene),numal,name)
              g2=getnam(-locus(i,gen2),numal,name)
            end if
          else
            ntyped=ntyped+1
            untyped(i)=.false.
            g1=getnam(locus(i,gene),numal,name)
            g2=getnam(locus(i,gen2),numal,name)
          end if
          call update(i,g1,g2,set)
   25   continue
        write(twrk) alltyp, pedigree, num, nfound, 
     &    (id(i),fa(i),mo(i), untyped(i), set(i,1), set(i,2), i=1, num)
      goto 5
   20 continue

      maxem=emiter
      if (nuntyp.eq.0) maxem=1

      den=1.0d0/dfloat(2*tfound)
C
C EM iterations
C
      do 99 i=1, numal
        alfrq(i)=1.0d0/dfloat(numal)
   99 continue

      emit=0
  100 continue
        emit=emit+1
        do 101 i=1, numal
          alfrq2(i)=0.0d0
  101   continue
        do 102 i=1,4
          proprate(i)=0
          proptyp(i)=0
  102   continue
        call genot(numal,alfrq,ngtp,gfrq)
        call newnam(wrknum, wrkfil)
        open(twrk2,file=wrkfil,form='unformatted')
        rewind(twrk)
        do 65 j=1, nfam
          read(twrk) alltyp, pedigree, num, nfound, 
     &      (id(i),fa(i),mo(i),untyped(i), set(i,1),set(i,2), i=1, num)
          if (.not.alltyp) then
            call tabmat(num,nfound,fa,mo,untyped,nummat,cntmat)
            if (plevel.gt.1) then
              write(*,'(/2a)') 
     &          'Metropolis simulation of pedigree ',pedigree 
            end if
            do 45 it=1,iter
              call drop(it,pedigree,num,nfound,id,fa,mo,nummat,
     2               cntmat,untyped,numal,gfrq,set,set2,sibd,
     3               key,iprop,plevel)
              call cntprop(iprop, proprate, proptyp)
              do 50 i=1, nfound
                alfrq2(set(i,1))=alfrq2(set(i,1))+wei
                alfrq2(set(i,2))=alfrq2(set(i,2))+wei
   50         continue
   45       continue
          else
            do 55 i=1, nfound
              alfrq2(set(i,1))=alfrq2(set(i,1))+one
              alfrq2(set(i,2))=alfrq2(set(i,2))+one
   55       continue
          end if
          write(twrk2) alltyp, pedigree, num, nfound, 
     &      (id(i),fa(i),mo(i),untyped(i), set(i,1),set(i,2), i=1, num)
   65   continue
        accel=1.0d0
        if (emit.le.(maxem/2)) then
          accel=dfloat(emit+1)/dfloat(emit)
        end if
        do 75 i=1, numal
           delta=alfrq(i)-den*alfrq2(i)
           alfrq(i)=alfrq(i)-accel*delta
           if (alfrq(i).lt.0.0d0) alfrq(i)=0.01d0
           if (alfrq(i).gt.1.0d0) alfrq(i)=0.99d0
   75   continue
        if (plevel.gt.0) then
          call wrprop(emit, proprate, proptyp)
          write(*,'(i4,1x,12(1x,f5.3):)') emit, (alfrq(i), i=1, numal)
        end if
        close(twrk,status='delete')
        close(twrk2,status='keep')
        open(twrk,file=wrkfil,form='unformatted')
      if (emit.lt.maxem .and. irupt.eq.0) goto 100
      close(twrk,status='delete')
      if (plevel.gt.0) write(*,*)
      totall=2*tfound
      typed=tfound-nuntyp
      return
      end
C end-of-mcfreq
C
C Increment counts of MCMC proposal type
C
      subroutine cntprop(iprop, proprate, proptyp)
      integer iprop 
      integer proprate(4), proptyp(4)
      proptyp(abs(iprop))=proptyp(abs(iprop))+1
      if (iprop.gt.0) then
        proprate(iprop)=proprate(iprop)+1
      end if
      return
      end
C end-of-cntprop
C
C Summary of MCMC proposals from drop()
C
      subroutine wrprop(ilabel, proprate, proptyp)
      integer ilabel
      integer proprate(4), proptyp(4)
      character*3 proplab(4)
      data proplab /'ibd','mut','swi','loc'/

      if (ilabel.gt.0) then
        write(*,'(a,i4,$)') 'DROP ', ilabel
      else
        write(*,'(/a,$)') 'MCMC proposals:'
      end if
      write(*,'(4(1x, a3, i7, 1x, f5.3))')
     2  (proplab(i), proptyp(i), 
     3   dfloat(proprate(i))/dfloat(max(1,proptyp(i))), i=1,4)
      return
      end
C end-of-wrprop
